package com.atlassian.ozymandias;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static java.lang.String.format;

/**
 * <pre>
 *
 * I met a traveller from an antique land
 * Who said: "Two vast and trunkless legs of stone
 * Stand in the desert. Near them on the sand,
 * Half sunk, a shattered visage lies, whose frown
 * And wrinkled lip and sneer of cold command
 * Tell that its sculptor well those passions read
 * Which yet survive, stamped on these lifeless things,
 * The hand that mocked them and the heart that fed.
 * And on the pedestal these words appear:
 *
 * `My name is Ozymandias, King of Plugins:
 * Look on your lack of try catches, ye mighty, and despair!'
 *
 * Nothing beside remains. Round the decay
 * Of that colossal wreck, boundless and bare
 * The lone and level sands stretch far away
 *
 *      - Percy Shelly, hanging one weekend with Ada Lovelace
 *
 * </pre>
 * <p/>
 * Traditionally, and especially inside JIRA, we have shown scant regard for plugin point exception handling.  The
 * purpose of this code is to provide some.
 * <p/>
 * This can be used to safely iterate a set of plugin points and run some callable code with each of the modules found.
 * <p/>
 * It can handle the exceptions is a standard manner and hence help tighten up our plugin point code inside Atlassian.
 * It can also apply shortcut logic so that the first exception terminates the iteration directly.
 */
public class SafePluginPointAccess
{
    private final PluginAccessor pluginAccessor;
    private final Logger log;

    /**
     * Constructs a safe plugin point access which gets its modules from the provided {@link PluginAccessor} and uses
     * the passed in logger in the case of exceptions
     *
     * @param pluginAccessor this provides plugin modules
     * @param log the logger to use
     */
    private SafePluginPointAccess(final PluginAccessor pluginAccessor, Logger log)
    {
        this.pluginAccessor = pluginAccessor;
        this.log = log;
    }

    /**
     * @return a builder of SafePluginPointAccess
     */
    public static Builder builder()
    {
        return new Builder();
    }

    /**
     * @param pluginAccessor the one to use
     * @return a builder of SafePluginPointAccess
     */
    public static Builder builder(PluginAccessor pluginAccessor)
    {
        return new Builder().setPluginAccessor(pluginAccessor);
    }

    /**
     * This will find all enabled module descriptors by class and invoke the passed in callback function for each module
     * found.
     * <p/>
     * This method uses the {@link PluginAccessor} to get all enabled module descriptors for the passed in
     * moduleDescriptorClass.
     *
     * @param moduleDescriptorClass the class of the enabled module descriptors to get
     * @param callback the callback for each module / module descriptor
     * @param <D> the class of module descriptor
     * @param <MT> the type of the module returned by a module descriptor
     * @param <RT> the return type of each forEnabled
     * @return a list fo the successful forEnabled return types as an overall result
     */
    public <MT, RT, D extends ModuleDescriptor<MT>> List<RT> forEnabled(Class<D> moduleDescriptorClass, PluginPointFunction<MT, RT> callback)
    {
        List<D> moduleDescriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(moduleDescriptorClass);
        return forDescriptors(moduleDescriptors, callback);
    }

    /**
     * This will iterate the module descriptors and invoke the passed in callback function for each module found.
     * <p/>
     *
     * @param moduleDescriptors the list of module descriptors to iterate
     * @param callback the callback for each module / module descriptor
     * @param <D> the class of module descriptor
     * @param <MT> the type of the module returned by a module descriptor
     * @param <RT> the return type of each forEnabled
     * @return a list fo the successful forEnabled return types as an overall result
     */
    public <MT, RT, D extends ModuleDescriptor<MT>> List<RT> forDescriptors(Iterable<D> moduleDescriptors, final PluginPointFunction<MT, RT> callback)
    {
        final List<RT> results = Lists.newArrayList();
        final PluginPointVisitor<MT> visitor = new PluginPointVisitor<MT>()
        {
            @Override
            public void visit(final ModuleDescriptor<MT> moduleDescriptor, final MT module)
            {
                RT result = callback.onModule(moduleDescriptor, module);
                results.add(result);
            }
        };

        visitDescriptors(moduleDescriptors, visitor);

        return results;
    }

    /**
     * This will find all enabled module descriptors by class and invoke the passed in visitor function for each module
     * found.
     * <p/>
     * This method uses the {@link PluginAccessor} to get all enabled module descriptors for the passed in
     * moduleDescriptorClass.
     *
     * @param moduleDescriptorClass the class of the enabled module descriptors to get
     * @param visitor the visitor for each module / module descriptor
     * @param <D> the class of module descriptor
     * @param <MT> the type of the module returned by a module descriptor
     */
    public <MT, D extends ModuleDescriptor<MT>> void visitEnabled(Class<D> moduleDescriptorClass, PluginPointVisitor<MT> visitor)
    {
        List<D> moduleDescriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(moduleDescriptorClass);
        visitDescriptors(moduleDescriptors, visitor);
    }

    /**
     * This will iterate the module descriptors and invoke the passed in visitor function for each module found.
     * <p/>
     *
     * @param moduleDescriptors the list of module descriptors to iterate
     * @param visitor the visitor for each module / module descriptor
     * @param <D> the class of module descriptor
     * @param <MT> the type of the module returned by a module descriptor
     */
    public <MT, D extends ModuleDescriptor<MT>> void visitDescriptors(Iterable<D> moduleDescriptors, PluginPointVisitor<MT> visitor)
    {
        for (D moduleDescriptor : moduleDescriptors)
        {
            //
            // in the wild experience has shown that null module descriptors can be returned.
            // Now this really really should not happen...but it does.  At least in JIRA.  So
            // since this code is all about belts and braces, we handle this anomaly and continue on
            //
            if (moduleDescriptor == null)
            {
                continue;
            }
            // get the module
            MT module;
            try
            {
                module = moduleDescriptor.getModule();
            }
            catch (RuntimeException e)
            {
                String msg = format("Unable to access module from module '%s' for descriptor class '%s' because of '%s - %s'.  Continuing to next module...",
                        completeKey(moduleDescriptor), getClassName(moduleDescriptor), getClassName(e), e.getMessage());
                logException(msg, e, log);
                continue;
            }
            // now invoke the call back function with this module
            try
            {
                visitor.visit(moduleDescriptor, module);
            }
            catch (RuntimeException e)
            {
                String msg = format("Unable to access module from module '%s' of type '%s' for descriptor class '%s' because of '%s - %s'.  Continuing to next module...",
                        completeKey(moduleDescriptor), getClassName(module), getClassName(moduleDescriptor), getClassName(e), e.getMessage());
                logException(msg, e, log);
            }
        }
    }

    /**
     * This is called to log the exception incident at warn level and the stack trace as debug level If you want to take
     * some other action or log in a different way then here is your chance.
     *
     * @param msg the message to log
     * @param e the exception that occurred
     * @param log the logger to use
     */
    protected void logException(final String msg, final RuntimeException e, final Logger log)
    {
        log.warn(msg);
        if (log.isDebugEnabled())
        {
            log.debug(msg, e);
        }
    }

    private <MT, D extends ModuleDescriptor<MT>> String completeKey(final D moduleDescriptor)
    {
        if (moduleDescriptor == null)
        {
            return "NULL";
        }
        return moduleDescriptor.getCompleteKey();
    }

    private String getClassName(final Object o)
    {
        if (o == null)
        {
            return "NULL";
        }
        if (o instanceof Class)
        {
            return ((Class) o).getName();
        }
        return o.getClass().getName();
    }

    /**
     * A builder of {@link SafePluginPointAccess} objects
     */
    @SuppressWarnings ("UnusedDeclaration")
    static class Builder
    {
        private PluginAccessor pluginAccessor = null;
        private Logger log = LoggerFactory.getLogger(SafePluginPointAccess.class);

        public Builder setPluginAccessor(final PluginAccessor pluginAccessor)
        {
            this.pluginAccessor = pluginAccessor;
            return this;
        }

        public Builder setLog(final Logger log)
        {
            this.log = log;
            return this;
        }

        public SafePluginPointAccess build()
        {
            return new SafePluginPointAccess(pluginAccessor, log);
        }
    }
}

