package com.atlassian.browsers;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Defines the browser version, OS and the executable path for each browser version.
 */
enum BrowserInstaller
{
    FIREFOX_WINDOWS_3_5(BrowserVersion.FIREFOX_3_5, OS.WINDOWS, "firefox.exe"),
    FIREFOX_LINUX_3_5(BrowserVersion.FIREFOX_3_5, OS.LINUX, "firefox-bin"),
    FIREFOX_LINUX_3_6(BrowserVersion.FIREFOX_3_6, OS.LINUX, "firefox-bin"),
    FIREFOX_OSX_3_5(BrowserVersion.FIREFOX_3_5, OS.OSX, "Contents/MacOS/firefox-bin"),
    FIREFOX_OSX_3_6(BrowserVersion.FIREFOX_3_6, OS.OSX, "Contents/MacOS/firefox-bin"),
    CHROME_LINUX_6(BrowserVersion.CHROME_6, OS.LINUX, "chrome"),
    CHROME_MAC_5(BrowserVersion.CHROME_5, OS.OSX, "Contents/MacOS/chrome-bin"),
    CHROME_MAC_7(BrowserVersion.CHROME_7, OS.OSX, "Contents/MacOS/chrome-bin");

    private static final Logger log = LoggerFactory.getLogger(BrowserInstaller.class);

    private final BrowserVersion browser;
    private final OS os;
    private final String binaryPath;

    // Map of OS to supported browser versions the first in the list should be the default browser.
    private final static Map<OS, ImmutableList<BrowserVersion>> BROWSERS = ImmutableMap.of(
            OS.WINDOWS, ImmutableList.<BrowserVersion>of(BrowserVersion.FIREFOX_3_5),
            OS.LINUX, ImmutableList.<BrowserVersion>of(BrowserVersion.FIREFOX_3_5, BrowserVersion.CHROME_6),
            OS.OSX, ImmutableList.<BrowserVersion>of(BrowserVersion.FIREFOX_3_5, BrowserVersion.FIREFOX_3_6, BrowserVersion.CHROME_5, BrowserVersion.CHROME_7)
    );

    BrowserInstaller(BrowserVersion browser, OS os, String binaryPath)
    {
        this.browser = browser;
        this.os = os;
        this.binaryPath = binaryPath;
    }


    public BrowserVersion getBrowser()
    {
        return browser;
    }

    public OS getOS()
    {
        return os;
    }

    public String getOsDirName()
    {
        return os == null ? OS.ALL : os.getName();
    }

    public String getBinaryPath()
    {
        return binaryPath;
    }

    private static BrowserInstaller findBrowserInstaller(BrowserVersion browserVersion, OS os)
    {
        for (BrowserInstaller installer : values())
        {
            if (installer.browser.equals(browserVersion) && installer.os.equals(os))
            {
                return installer;
            }
        }

        log.info("Unable to find browser installer for browser {} on OS {}", browserVersion, os);

        return null;
    }

    /**
     * Determines the BrowserInstaller based on the browserStr passed and the current OS.
     * @param browserStr the browserStr that defines the BrowserInstaller that is needed.
     * @return The browser installer if it exists otherwise null.
     */
    public static BrowserInstaller typeOf(String browserStr)
    {
        OS os = OS.getType();

        BrowserType browserType = BrowserType.typeOf(browserStr);
        BrowserVersion browserVer = BrowserVersion.typeOf(browserStr);

        // Just get the first browser that is supported by the OS.
        if (browserVer == null && browserType != null && browserType.getName().equals(browserStr))
        {
            List<BrowserVersion> browserVersions = new ArrayList<BrowserVersion>();

            for (BrowserVersion browserVersion : BROWSERS.get(os))
            {
                if (browserVersion.getBrowser().equals(browserType))
                {
                    browserVersions.add(browserVersion);
                }
            }

            if (browserVersions.size() > 0)
            {
                Collections.sort(browserVersions);
                // Get the latest version of the browser for the os.
                return findBrowserInstaller(browserVersions.get(browserVersions.size()-1), os);
            }

            log.info("No installer for browser {}", browserType);
        }

        else if (browserVer != null)
        {
            if (!BrowserInstaller.isBrowserSupportedForOs(browserVer))
            {
                throw new RuntimeException(String.format("Unsupported browser {} for OS {}", browserVer, os));
            }
            else
            {
                return findBrowserInstaller(browserVer, os);
            }
        }

        log.info("There are no browser installers for browser string {}", browserStr);

        return null;
    }

    public static boolean isBrowserSupportedForOs(BrowserVersion browserVersion)
    {
        OS os = OS.getType();
        List<BrowserVersion> supportedOsBrowsers = BROWSERS.get(os);

        return supportedOsBrowsers.contains(browserVersion);
    }

    /**
     * Installs the current browser into the destination directory specified by
     * extracting the browser zip file.
     * If there is a profile zip for the browser it will also extract this.
     * Then the correct permissions are applied to the required files.
     * @param destDir The location to extract the browser into. This is the parent directory for the browser.
     * @param installConfigurator 
     */
    public void install(File destDir, DefaultBrowserInstallConfigurator installConfigurator)
    {

        String browserName = browser.getBrowserName();
        String binaryPath = getBinaryPath();
        String osDirName = getOsDirName();
        String profileName = browserName + "-profile";

        String browserResource = "/" + osDirName + "/" + browserName + ".zip";
        String profileResource = "/" + osDirName + "/" + profileName + ".zip";

        File browserProfile = null;

        try {
            File browserDir = Utils.extractZip(destDir, browserResource);

            if (Utils.resourceExists(profileResource))
            {
                browserProfile = Utils.extractZip(destDir, profileResource);
            }

            File browserBinary = new File(browserDir, binaryPath);

            if (!OsValidator.isWindows())
            {
                Utils.make755(browserBinary);
            }

            BrowserConfig browserConfig = new BrowserConfig(browserDir, browserBinary, browserProfile);

            installConfigurator.setupBrowser(browser, browserConfig);

        } catch(IOException e)
        {
            throw new RuntimeException(e);
        }
    }
}

