package at.jku.isse.gradient.gradle.tasks

import at.jku.isse.gradient.Gradient
import at.jku.isse.gradient.model.StructuralCache
import at.jku.isse.gradient.service.CodeService
import com.google.gson.Gson
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.io.FileWriter
import javax.inject.Inject


open class CodeAnalysisTask
@Inject constructor(@Input private val groupName: Property<String>,
                    @Input private val projectName: Property<String>,
                    @Input private val gradientModelRegex: Property<String>,
                    @InputFiles val inputFiles: FileCollection,
                    @InputFiles val classFiles: FileCollection,
                    @OutputFile val gradientCache: File) : DefaultTask() {

    private val gradient = Gradient()
    private val codeService: CodeService

    init {
        group = "analysis"
        description = "Executes the Gradient code analysis and sets the information to the Gradient Server"

        logger.debug("Bootstrapping gradient client.")
        val injector = gradient.bootstrapGradient()
        codeService = injector.getInstance(CodeService::class.java)

        setOnlyIf {
            gradient.gradientBackendAvailable()
        }
    }

    private fun executeAnalysis(): StructuralCache {
        require(!inputFiles.isEmpty) { "Expected non-empty input files." }

        val inputPaths = inputFiles.map { it.absolutePath }
        val classPaths = classFiles.map { it.absolutePath }

        logger.debug("Staring the code analysis: ${inputPaths.size} source files")

        val additionalGradientModelRegex = if (gradientModelRegex.isPresent) listOf(gradientModelRegex.get()) else listOf()
        val codeModel = codeService.analyze(groupName.get(), projectName.get(), inputPaths, classPaths, additionalGradientModelRegex)

        return codeService.prepareGradientCache(codeModel)
                .also { it.additionalGradientModelRegex = gradientModelRegex.orNull }
    }

    @TaskAction
    fun analyze() {
        val cache = executeAnalysis()

        logger.debug("Writing gradient cache.")
        FileWriter(gradientCache).use {
            Gson().toJson(cache, it)
        }
    }
}