package at.jku.isse.gradient.gradle

import at.jku.isse.gradient.gradle.tasks.CompileAspectjTask
import at.jku.isse.gradient.gradle.tasks.GenerateAdditionalPointcutsTask
import at.jku.isse.gradient.gradle.tasks.CodeAnalysisTask
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.logging.LogLevel
import org.gradle.api.logging.Logger
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.JavaPluginConvention
import org.gradle.api.tasks.compile.AbstractCompile


@Suppress("unused")
class GradientGradlePlugin : Plugin<Project> {

    private lateinit var logger: Logger

    override fun apply(project: Project) {

        with(project) {
            plugins.withType(JavaPlugin::class.java) {
                this@GradientGradlePlugin.logger = logger
                logging.captureStandardOutput(LogLevel.DEBUG)
                logging.captureStandardError(LogLevel.ERROR)
                configureExtensions(project)
                configureConfigurations(project)
                configureRepositories(project)
                configureDependencies(project)
                configureDynamicPointcutsTask(project)
                configureCompileAspectTask(project)
                configureCodeAnalysisTask(project)
            }
        }
    }

    private fun configureExtensions(project: Project) {
        project.extensions.create("gradient", GradientExtension::class.java, project)
    }

    private fun configureConfigurations(project: Project) {
        with(project) {
            logger.debug("Creating configurations")

            configurations.create(Conventions.CONFIGURATION_ASPECTJ)
            configurations.create(Conventions.CONFIGURATION_GRADIENT)
        }
    }

    private fun configureRepositories(project: Project) {
        logger.debug("Configuring repository: jcenter")

        project.repositories.jcenter()
    }

    private fun configureDependencies(project: Project) {
        with(project) {
            logger.debug("Configuring dependencies")

            val aspectjConfig = configurations.getByName(Conventions.CONFIGURATION_ASPECTJ)
            aspectjConfig.dependencies.add(dependencies.create(Conventions.dependencyAspectjTools))
            aspectjConfig.dependencies.add(dependencies.create(Conventions.dependencyAspectjRt))

            val gradientConfig = configurations.getByName(Conventions.CONFIGURATION_GRADIENT)
            gradientConfig.dependencies.add(dependencies.create(Conventions.dependencyGradientAspects))

            val runtimeConfig = configurations.getByName(Conventions.CONFIGURATION_RUNTIME_CLASSPATH)
            runtimeConfig.dependencies.add(dependencies.create(Conventions.dependencyAspectjRt))
            runtimeConfig.dependencies.add(dependencies.create(Conventions.dependencyGradientAspects))

            val compileConfig = configurations.getByName("compile")
            compileConfig.dependencies.add(dependencies.create(Conventions.dependencyGradientAspects))
        }
    }

    private fun configureDynamicPointcutsTask(project: Project) {
        with(project) {
            logger.debug("Configuring generateDynamicPointcuts tasks")

            val extension = extensions.getByType(GradientExtension::class.java)

            tasks.create(
                    Conventions.TASK_GENERATE_ADDITIONAL_POINTCUTS,
                    GenerateAdditionalPointcutsTask::class.java,
                    layout.projectDirectory.dir(Conventions.FILE_ADDITIONAL_POINTCUTS),
                    extension.includes
            )
        }
    }

    private fun configureCompileAspectTask(project: Project) {
        with(project) {
            logger.debug("Configuring compileAspect task")

            val aspectjConfiguration = configurations.getByName(Conventions.CONFIGURATION_ASPECTJ)
            val gradientConfiguration = configurations.getByName(Conventions.CONFIGURATION_GRADIENT)
            val compileConfiguration = configurations.getByName(Conventions.CONFIGURATION_COMPILE_CLASSPATH)

            val compileJavaTask = tasks.getByName(Conventions.TASK_COMPILE_JAVA) as AbstractCompile

            val compileAspectTask = tasks.create(
                    Conventions.TASK_COMPILE_ASPECTJ,
                    CompileAspectjTask::class.java,
                    project.files(), //srcDir
                    compileJavaTask.outputs.files, // inputPaths
                    gradientConfiguration.fileCollection { true }, // aspectPaths
                    project.files(aspectjConfiguration.fileCollection { true },
                            compileConfiguration.fileCollection { true }), // classPaths
                    compileJavaTask.outputs.files) // destinationDirectories

            compileJavaTask.finalizedBy(compileAspectTask)
            compileAspectTask.dependsOn(Conventions.TASK_GENERATE_ADDITIONAL_POINTCUTS)
        }
    }

    private fun configureCodeAnalysisTask(project: Project) {
        with(project) {
            logger.debug("Configuring code analysis task.")

            val sourceSets = convention.getPlugin(JavaPluginConvention::class.java).sourceSets

            val extension = extensions.getByType(GradientExtension::class.java)
            val compileConfiguration = configurations.getByName(Conventions.CONFIGURATION_COMPILE_CLASSPATH)

            tasks.create(
                    Conventions.TASK_CODE_ANALYSIS,
                    CodeAnalysisTask::class.java,
                    extension.groupName,
                    extension.projectName,
                    extension.includes,
                    sourceSets.getByName("main").allJava.sourceDirectories,
                    compileConfiguration.fileCollection { true },
                    file("src/main/resources/gcache.json")
            )
        }
    }
}