package at.jku.isse.gradient.gradle.tasks

import at.jku.isse.gradient.Gradient
import at.jku.isse.gradient.model.StructuralCache
import at.jku.isse.gradient.service.StructuralService
import com.google.gson.Gson
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.io.FileWriter
import javax.inject.Inject


open class StructuralAnalysisTask
@Inject constructor(@Input private val groupName: Property<String>,
                    @Input private val projectName: Property<String>,
                    @Input private val gradientModelRegex: Property<String>,
                    @InputFiles val inputFiles: FileCollection,
                    @InputFiles val classFiles: FileCollection,
                    @OutputFile val structuralCacheFile: File) : DefaultTask() {

    private val gradient = Gradient()
    private val structuralService: StructuralService

    init {
        group = "analysis"
        description = "Executes the Gradient structural analysis and sets the information to the Gradient Server"

        logger.debug("Bootstrapping gradient client.")
        val injector = gradient.bootstrapGradient()
        structuralService = injector.getInstance(StructuralService::class.java)

        setOnlyIf {
            gradient.gradientBackendAvailable()
        }
    }

    private fun executeAnalysis(): StructuralCache {
        require(!inputFiles.isEmpty) { "Expected non-empty input files." }

        val inputPaths = inputFiles.map { it.absolutePath }
        val classPaths = classFiles.map { it.absolutePath }

        logger.debug("Staring the structural analysis: ${inputPaths.size} source files")

        val additionalGradientModelRegex = if (gradientModelRegex.isPresent) listOf(gradientModelRegex.get()) else listOf()
        val structuralModel = structuralService.analyze(groupName.get(), projectName.get(), inputPaths, classPaths, additionalGradientModelRegex)

        return structuralService.prepareStructuralCache(structuralModel)
                .also { it.additionalGradientModelRegex = gradientModelRegex.orNull }
    }

    @TaskAction
    fun analyze() {
        val structuralCache = executeAnalysis()

        logger.debug("Writing structural cache.")
        FileWriter(structuralCacheFile).use {
            Gson().toJson(structuralCache, it)
        }
    }
}