package at.jku.isse.gradient.gradle.tasks

import at.jku.isse.gradient.Gradient
import at.jku.isse.gradient.service.StructuralService
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.io.FileWriter
import java.util.*
import javax.inject.Inject


open class StructuralAnalysisTask
@Inject constructor(@Input private val groupName: Property<String>,
                    @Input private val projectName: Property<String>,
                    @InputFiles val inputFiles: FileCollection,
                    @InputFiles val classFiles: FileCollection,
                    @OutputFile val structuralPropertiesFile: File) : DefaultTask() {

    private val gradient = Gradient()
    private val structuralService: StructuralService

    init {
        group = "analysis"
        description = "Executes the Gradient structural analysis and sets the information to the Gradient Server"

        logger.debug("Bootstrapping gradient client.")
        val injector = gradient.bootstrapGradient()
        structuralService = injector.getInstance(StructuralService::class.java)

        setOnlyIf {
            gradient.gradientBackendAvailable()
        }
    }

    @TaskAction
    fun analyze() {
        require(!inputFiles.isEmpty) { "Expected non-empty input files." }

        val inputPaths = inputFiles.map { it.absolutePath }
        val classPaths = classFiles.map { it.absolutePath }

        logger.debug("Staring the structural analysis: ${inputPaths.size} source files")
        val project = structuralService.analyze(groupName.get(), projectName.get(), inputPaths, classPaths)

        logger.debug("Writing structural cache.")
        val prop = Properties()
        prop.setProperty("project.name", project.canonicalName)
        prop.setProperty("project.version", project.version.id.toString())
        prop.store(FileWriter(structuralPropertiesFile), null)
    }
}