package at.jku.isse.gradient.gradle.tasks

import at.jku.isse.gradient.gradle.Conventions
import org.apache.tools.ant.BuildException
import org.apache.tools.ant.Project
import org.apache.tools.ant.types.Path
import org.aspectj.tools.ant.taskdefs.AjcTask
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileCollection
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.nio.file.Files
import javax.inject.Inject

open class CompileAspectjTask @Inject constructor(
        /**
         * Directories containing source files (ending with .java or .aj) to compile.
         */
        @InputFiles val srcDir: FileCollection,
        /**
         * Read .class files for bytecode weaving from directories or zip files (like classpath).
         */
        @InputFiles val inputPaths: FileCollection,
        /**
         * Similar to classpath, aspectpath contains read-only, binary aspect libraries that are woven into sources but not included in the output. aspectpath accepts jar/zip files (but, unlike classpath, not directories).
         */
        @InputFiles val aspectPaths: FileCollection,
        /**
         * The classpath used by the sources being compiled. When compiling aspects, include the same version of the aspectjrt.jar.
         */
        @InputFiles val classPaths: FileCollection,
        /**
         * The directory in which to place the generated class files. Only one of destDir and outJar may be set
         */
        @InputFiles val destinationDirectories: FileCollection) : DefaultTask() {

    companion object {
        private val exists = { it: File -> it.exists() }
    }

    init {
        description = "Weaves the aspects into the Bytecode using the Aspectj compiler."
    }

    @TaskAction
    fun compile() {
        val hasSourceFiles = inputPaths.files.any {
            it.exists() && Files.walk(it.toPath()).anyMatch { Files.isRegularFile(it) }
        }

        if (hasSourceFiles) {
            assert(classPaths.files.any { it.name.contains("aspectjrt") }) { "Aspectj runtime not present" }

            logger.info("Compiling with ajc.")
            logger.debug("Configuring aspectj ant task.")

            logger.debug("srcDir ${srcDir.files}")
            logger.debug("inputPaths ${inputPaths.files}")
            logger.debug("aspectPaths ${aspectPaths.files}")
            logger.debug("classpaths ${classPaths.files}")
            logger.debug("destinationDirectories ${destinationDirectories.files}")

            with(AjcTask()) {
                project = Project()
                project.baseDir = this@CompileAspectjTask.project.projectDir

                setSource(Conventions.JAVA_SOURCE_COMPATIBILITY)
                setTarget(Conventions.JAVA_TARGET_COMPATIBILITY)
                srcDir.filter(exists)
                        .forEach { setSrcDir(Path(project, it.path)) }
                inputPaths.filter(exists)
                        .forEach { setInpath(Path(project, it.path)) }
                aspectPaths.filter(exists)
                        .forEach { setAspectpath(Path(project, it.path)) }
                classPaths.filter(exists)
                        .forEach { setClasspath(Path(project, it.path)) }
                destinationDirectories
                        .forEach { setDestdir(it) }

                try {
                    logger.debug("Starting compilation")

                    execute()

                    logger.debug("Finished compilation")
                } catch (ex: BuildException) {
                    logger.error("Compiling the sources with the aspectj compiler failed.", ex)
                }
            }
        } else {
            logger.warn("Skipping aspectj compilation as neither class files not source files are available to compile.")
        }
    }
}
