package at.jku.isse.gradient.gradle

import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.logging.Logger
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.JavaPluginConvention
import org.gradle.api.tasks.compile.AbstractCompile


open class GradientGradlePlugin : Plugin<Project> {

    private lateinit var logger: Logger

    override fun apply(project: Project) {
        with(project) {
            plugins.withType(JavaPlugin::class.java) {
                this@GradientGradlePlugin.logger = logger

                configureExtensions(project)
                configureRepositories(project)
                configureConfigurations(project)
                configureDependencies(project)
                configureCompileAspectTask(project)
                configureStructuralAnalysisTask(project)
            }
        }
    }

    private fun configureExtensions(project: Project) {
        project.extensions.create("gradient", GradientExtension::class.java, project)
    }

    private fun configureRepositories(project: Project) {
        with(project) {
            repositories.mavenCentral()
            repositories.maven {
                it.setUrl(Conventions.REPOSITORIES_BINTRAY_TALLIC)
            }
        }
    }

    private fun configureConfigurations(project: Project) {
        with(project) {
            logger.debug("Creating configurations")

            configurations.create(Conventions.CONFIGURATION_ASPECTJ)
            configurations.create(Conventions.CONFIGURATION_GRADIENT)
        }
    }

    private fun configureDependencies(project: Project) {
        with(project) {
            val aspectjConfig = configurations.getByName(Conventions.CONFIGURATION_ASPECTJ)
            aspectjConfig.dependencies.add(dependencies.create(Conventions.dependencyAspectjTools))
            aspectjConfig.dependencies.add(dependencies.create(Conventions.dependencyAspectjRt))

            val gradientConfig = configurations.getByName(Conventions.CONFIGURATION_GRADIENT)
            gradientConfig.dependencies.add(dependencies.create(Conventions.dependencyGradientAspects))
        }
    }

    private fun configureCompileAspectTask(project: Project) {
        with(project) {
            logger.debug("Configuring compileAspect task")

            val aspectConfiguration = configurations.getByName(Conventions.CONFIGURATION_ASPECTJ)
            val gradientConfiguration = configurations.getByName(Conventions.CONFIGURATION_GRADIENT)
            val compileConfiguration = configurations.getByName(Conventions.CONFIGURATION_COMPILE_CLASSPATH)

            val compileJavaTask = tasks.getByName(Conventions.TASK_COMPILE_JAVA) as AbstractCompile

            val compileAspectTask = tasks.create(Conventions.TASK_COMPILE_ASPECTJ, CompileAspectj::class.java) {
                it.inputPaths.from(compileJavaTask.destinationDir)
                afterEvaluate { _ ->
                    it.aspectPaths.from(gradientConfiguration.files)
                    it.classPaths.from(aspectConfiguration.files)
                    it.classPaths.from(compileConfiguration.files)
                }
                it.destinationDirectories.from(compileJavaTask.destinationDir)
            }

            compileJavaTask.finalizedBy(compileAspectTask)
        }
    }

    private fun configureStructuralAnalysisTask(project: Project) {

        with(project) {
            logger.debug("Configuring structural analysis task.")

            val sourceSets = convention.getPlugin(JavaPluginConvention::class.java).sourceSets
            val configuration = extensions.getByType(GradientExtension::class.java)

            val structuralAnalysisTask = tasks.create(
                    Conventions.TASK_STRUCTURAL_ANALYSIS,
                    StructuralAnalysis::class.java,
                    configuration.groupName,
                    configuration.projectName,
                    sourceSets.getByName("main").allJava.sourceDirectories
            )

            tasks.getByName("compileJava").finalizedBy(structuralAnalysisTask)
        }
    }
}