package at.jku.isse.gradient.gradle

import at.jku.isse.gradient.Gradient
import at.jku.isse.gradient.monitoring.MonitoringModule
import at.jku.isse.gradient.service.StructuralService
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.TaskAction

open class StructuralAnalysis @javax.inject.Inject constructor(@Input private val groupName: Property<String>,
                                                               @Input private val projectName: Property<String>,
                                                               @InputFiles private val inputFiles: FileCollection) : DefaultTask() {

    private val structuralService: StructuralService? by lazy {
        setupGradient()
    }

    init {
        group = "analysis"
        description = "Executes the Gradient structural analysis and sets the information to the Gradient Server"

        setOnlyIf {
            structuralService != null
        }
    }

    private fun setupGradient(): StructuralService? {
        logger.debug("Bootstrapping gradient client.")
        return try {
            val gradient = Gradient()
            gradient.registerAdditionalModules(listOf(MonitoringModule()))
            gradient.bootstrapGradient().getInstance(StructuralService::class.java)
        } catch (e: Exception) {
            logger.info("Structural Analysis will be skipped as not backend service is available.")
            logger.debug("Error during bootstrapping the gradient client library", e)
            null
        }
    }

    @TaskAction
    fun analyze() {
        require(!inputFiles.isEmpty) { "Expected non-empty input files." }

        structuralService?.let {
            val inputPaths = inputFiles.map { it.absolutePath }
            it.analyze(groupName.get(), projectName.get(), inputPaths)
        }
    }
}