package at.jku.isse.psma.gradient.gradle

import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.logging.Logger
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.JavaPluginConvention
import org.gradle.api.tasks.compile.AbstractCompile


open class GradientGradlePlugin : Plugin<Project> {

    private lateinit var logger: Logger

    override fun apply(project: Project) {
        with(project) {
            plugins.withType(JavaPlugin::class.java, {
                this@GradientGradlePlugin.logger = logger

                configureExtensions(project)
                configureConfigurations(project)
                configureRepositories(project)
                configureDependencies(project)
                configureSourceSets(project)
                configureCompileAspectTask(project)
                configureStructuralAnalysisTask(project)
            })
        }
    }

    private fun configureExtensions(project: Project) {
        project.extensions.create("gradient", GradientExtension::class.java, project)
    }

    private fun configureConfigurations(project: Project) {
        with(project) {
            logger.debug("Creating aspectj configurations")

            configurations.create(CONFIGURATION_ASPECT)
        }
    }

    private fun configureRepositories(project: Project) {
        with(project) {
            logger.debug("Configuring additional repositories")

            repositories.mavenCentral()
        }
    }

    private fun configureDependencies(project: Project) {
        with(project) {
            afterEvaluate {
                logger.debug("Configuring dependencies")

                dependencies.add(CONFIGURATION_RUNTIME_ONLY, DEPENDENCY_ASPECTJRT)
            }
        }
    }

    private fun configureSourceSets(project: Project) {
        with(project) {
            logger.debug("Configuring source sets")
            val sourceSets = convention.getPlugin(JavaPluginConvention::class.java).sourceSets
            val mainSourceSet = sourceSets.getByName(SOURCESET_INPUT_JAVA)
            mainSourceSet.allSource.srcDirs(SOURCE_PATH_ASPECT)
        }
    }

    private fun configureCompileAspectTask(project: Project) {
        with(project) {
            logger.debug("Configuring compileAspect task")

            val aspectConfiguration = configurations.getByName(CONFIGURATION_ASPECT)

            val compileJavaTask = tasks.getByName(TASK_COMPILE_JAVA) as AbstractCompile
            val compileKotlinTask = tasks.findByName(TASK_COMPILE_KOTLIN) as AbstractCompile?

            val compileAspectTask = tasks.create(TASK_COMPILE_ASPECTJ, CompileAspectj::class.java, {
                it.sourceRoots.setFrom(SOURCE_PATH_ASPECT)
                it.inputPaths.setFrom(files(compileJavaTask.destinationDir))
                compileKotlinTask?.let { task -> it.inputPaths.from(task) }
                it.aspectPaths.setFrom(aspectConfiguration.files)
                it.classPaths.setFrom(compileJavaTask.classpath)
                it.destinationDirectories.setFrom(compileJavaTask.destinationDir)
            })

            compileJavaTask.finalizedBy(compileAspectTask)
        }
    }

    private fun configureStructuralAnalysisTask(project: Project) {

        with(project) {
            logger.debug("Configuring structural analysis task.")

            val sourceSets = convention.getPlugin(JavaPluginConvention::class.java).sourceSets
            val configuration = extensions.getByType(GradientExtension::class.java)

            val structuralAnalysisTask = tasks.create(
                    TASK_STRUCTURAL_ANALYSIS,
                    StructuralAnalysis::class.java,
                    configuration.groupName,
                    configuration.projectName,
                    sourceSets.getByName("main").allJava.sourceDirectories
            )

            tasks.getByName("build").finalizedBy(structuralAnalysisTask)
        }
    }
}