package at.jku.isse.gradient.lang.java

import at.jku.isse.gradient.model.DataType
import com.google.common.util.concurrent.AtomicDouble
import com.google.common.util.concurrent.AtomicDoubleArray
import spoon.reflect.code.CtFieldAccess
import spoon.reflect.code.CtInvocation
import spoon.reflect.declaration.*
import spoon.reflect.reference.*
import java.io.File
import java.net.URL
import java.nio.file.Path
import java.util.*
import java.util.concurrent.atomic.*
import java.util.logging.Level
import java.util.regex.Pattern

internal fun CtTypeReference<*>.canonicalName(): String {
    return when (this) {
        is CtArrayTypeReference<*> -> this.componentType.canonicalName()
        is CtTypeParameterReference -> canonicalName()
        else -> box().qualifiedName
    }
}

internal fun CtExecutableReference<*>.canonicalName(): String {
    val declaringType = this.declaringType.canonicalName()
    val executableName = if (this.isConstructor) this.declaringType.simpleName else this.simpleName

    val parameters = this.parameters.joinToString(", ") { parameter ->
        if (parameter.isGenerics && parameter is CtTypeParameterReference) {
            parameter.canonicalName()
        } else {
            parameter.box().qualifiedName
        }
    }

    return "$declaringType#$executableName($parameters)"
}

internal fun CtTypeParameterReference.canonicalName(): String {
    val boundingType = if (this.boundingType == null) "" else ": ${this.boundingType.canonicalName()}"

    return "${this.qualifiedName} $boundingType"
}

internal fun CtElement.canonicalName(): String {
    return when (this) {
        is CtTypeReference<*> -> this.canonicalName()
        is CtExecutableReference<*> -> this.canonicalName()
        is CtTypeParameterReference -> this.canonicalName()
        is CtParameter<*> -> "${this.parent.reference.canonicalName()} <- ${this.simpleName}"
        is CtFieldAccess<*> -> "${this.target} -> ${this.variable}"
        is CtInvocation<*> -> "${this.target} -> ${this.executable}"
        else -> this.toString()
    }
}

abstract class BaseTransformer(private val declarationCache: DeclarationCache) {

    companion object {
        internal fun CtTypeReference<*>.toDataType(): DataType {
            val typeFactory = this.factory.Type()

            return when {
                this.isGenerics && !this.isClass && !this.isInterface -> DataType.UNKNOWN
                this is CtTypeParameterReference && this.boundingType == null -> DataType.UNKNOWN
                this is CtTypeParameterReference && this.boundingType != null -> this.boundingType.toDataType()
                this.box().isSubtypeOf(typeFactory.VOID) -> DataType.VOID
                this.box().isSubtypeOf(typeFactory.BOOLEAN) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.BYTE) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.SHORT) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.INTEGER) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.LONG) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.FLOAT) -> DataType.FLOAT
                this.box().isSubtypeOf(typeFactory.DOUBLE) -> DataType.FLOAT
                this is CtArrayTypeReference<*> -> this.componentType.box().toDataType()
                this.box().isSubtypeOf(typeFactory.get<AtomicBoolean>(AtomicBoolean::class.java).reference) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.get<AtomicBoolean>(AtomicInteger::class.java).reference) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.get<AtomicBoolean>(AtomicLong::class.java).reference) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.get<AtomicBoolean>(AtomicDouble::class.java).reference) -> DataType.FLOAT
                this.isSubtypeOf(typeFactory.get<AtomicIntegerArray>(AtomicIntegerArray::class.java).reference) -> DataType.INT
                this.isSubtypeOf(typeFactory.get<AtomicLongArray>(AtomicLongArray::class.java).reference) -> DataType.INT
                this.isSubtypeOf(typeFactory.get<AtomicDoubleArray>(AtomicDoubleArray::class.java).reference) -> DataType.FLOAT
                this.isSubtypeOf(typeFactory.get<Date>(Date::class.java).reference) -> DataType.INT
                this.box().isSubtypeOf(typeFactory.CHARACTER) -> DataType.TEXT
                this.box().isSubtypeOf(typeFactory.STRING) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<CharSequence>(CharSequence::class.java).reference.box()) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<File>(File::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Path>(Path::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<URL>(URL::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Pattern>(Pattern::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Level>(Level::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Class<*>>(Class::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Enum<*>>(Enum::class.java).reference) -> DataType.TEXT
                this.isSubtypeOf(typeFactory.get<Map.Entry<*, *>>(Map.Entry::class.java).reference) && this.actualTypeArguments.size > 1 -> DataType.TEXT
                this.isSubtypeOf(typeFactory.MAP) && this.actualTypeArguments.size > 1 -> this.actualTypeArguments[1].toDataType()
                this.isSubtypeOf(typeFactory.ITERABLE) && this.actualTypeArguments.isNotEmpty() -> this.actualTypeArguments.first().toDataType()
                else -> DataType.UNKNOWN
            }
        }
    }

    internal fun CtTypeReference<*>.safeDeclaration(): Optional<CtType<*>> {
        return declarationCache.type.get(this)
    }

    internal fun CtFieldReference<*>.safeDeclaration(): Optional<CtField<*>> {
        return declarationCache.property.get(this)
    }

    internal fun CtExecutableReference<*>.safeDeclaration(): Optional<CtExecutable<*>> {
        return declarationCache.executable.get(this)
    }

    internal fun CtParameterReference<*>.safeDeclaration(): Optional<CtParameter<*>> {
        return declarationCache.parameter.get(this)
    }
}