package at.jku.isse.gradient.lang.java

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.Util
import at.jku.isse.gradient.annotations.GradientModel
import at.jku.isse.gradient.model.*
import at.jku.isse.gradient.profiledTrace
import com.google.common.hash.HashCode
import mu.KotlinLogging
import spoon.reflect.declaration.*
import spoon.reflect.reference.CtExecutableReference
import spoon.reflect.reference.CtFieldReference
import spoon.reflect.reference.CtReference
import spoon.reflect.reference.CtTypeReference

private val logger = KotlinLogging.logger {}

/**
 * Collects all entity, i.e., all explicitly modeled classes without the references among them.
 */
class AsgNodeTransformer(declarationCache: DeclarationCache) : BaseTransformer(declarationCache) {


    fun transform(ctTypeReference: CtTypeReference<*>, additionalGradientModelRegex: Collection<String>): StructuralModel {
        assert(additionalGradientModelRegex.all { it.isNotBlank() })

        val modelComponents = StructuralModel()
        try {

            logger.profiledTrace { "Transforming ${ctTypeReference.canonicalName()}" }

            val transformer = Transformer(ctTypeReference, modelComponents, additionalGradientModelRegex)
            transformer.scan(ctTypeReference)

            logger.profiledTrace(ProfilerState.STOP) { "Finished transforming ${ctTypeReference.canonicalName()}" }

            return transformer.structuralModel
        } catch (ex: Exception) {
            logger.error(ex) { "Error while transforming type: ${ctTypeReference.qualifiedName}" }
        }

        return modelComponents
    }

    private fun versionHash(reference: CtReference): HashCode {
        var hashCode = hash(reference.canonicalName())

        when (reference) {
            is CtTypeReference<*> -> {
                reference.safeDeclaration().ifPresent {
                    if (!it.isShadow) {
                        hashCode = hash(it.toString())
                    }
                }
            }
            is CtFieldReference<*> -> {
                reference.safeDeclaration().ifPresent {
                    if (!it.isShadow) {
                        hashCode = hash(it.toString())
                    }
                }
            }
            is CtExecutableReference<*> -> {
                reference.safeDeclaration().ifPresent {
                    if (it is CtShadowable && !it.isShadow) {
                        hashCode = hash(it.toString())
                    }
                }
            }
        }

        return hashCode
    }

    private fun versionHash(reference: CtTypeParameter): HashCode {
        return hash(reference.canonicalName())
    }

    private inner class Transformer(val thisType: CtTypeReference<*>,
                                    val structuralModel: StructuralModel = StructuralModel(),
                                    gradientModelRegex: Collection<String>) : SpoonVisitor() {

        private val gradientModelMatcher: List<Regex> = gradientModelRegex.map { Regex(it) }
        private val types: MutableMap<String, Type> by structuralModel.components
        private val properties: MutableMap<String, Property> by structuralModel.components
        private val executables: MutableMap<String, Executable> by structuralModel.components
        private val parameters: MutableMap<String, Parameter> by structuralModel.components

        private var currentExecutable: Executable? = null

        private val visitedElements = mutableSetOf<String>()

        override fun scan(element: CtElement?) {
            if (element == null) return

            val elementName = element.canonicalName()
            if (!visitedElements.contains(elementName)) {
                logger.trace { "Visiting: $elementName" }
                visitedElements.add(elementName)
                super.scan(element)
            }
        }

        override fun <T : Any?> visitCtTypeReference(r: CtTypeReference<T>) {
            assert(r == thisType)
            val versionHash = versionHash(r)
            val type = Type(
                    id = Util.uuid(),
                    name = r.simpleName,
                    canonicalName = r.canonicalName(),
                    availability = Availability.UNAVAILABLE,
                    quality = Quality.CONCRETE,
                    visibility = Visibility.PUBLIC,
                    isGenerics = false,
                    isGradientModel = false,
                    versionHash = versionHash
            ).also { types[it.canonicalName] = it }

            r.safeDeclaration().ifPresent { ctType ->
                type.run {
                    availability = if (r.declaration != null) Availability.SOURCE else Availability.BINARY
                    quality = when {
                        r.isInterface -> Quality.PURE
                        ModifierKind.ABSTRACT in r.modifiers -> Quality.PARTIAL
                        else -> Quality.CONCRETE
                    }
                    visibility = when {
                        ModifierKind.PRIVATE in r.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in r.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    }
                    isModel = ctType.getAnnotation(GradientModel::class.java) != null || gradientModelMatcher.any { it.matches(type.canonicalName) }
                }
                scan(ctType.allFields)
                scan(ctType.allExecutables)
                scan(ctType.formalCtTypeParameters)
            }
        }

        override fun visitCtTypeParameter(t: CtTypeParameter) {
            val versionHash = versionHash(t)
            Type(
                    id = Util.uuid(),
                    name = t.simpleName,
                    canonicalName = t.reference.canonicalName(),
                    availability = Availability.UNAVAILABLE,
                    quality = Quality.PURE,
                    visibility = when {
                        ModifierKind.PRIVATE in t.typeParameterDeclarer.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in t.typeParameterDeclarer.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    },
                    isGenerics = true,
                    isGradientModel = false,
                    versionHash = versionHash
            ).also { types[it.canonicalName] = it }
        }

        override fun <T : Any?> visitCtFieldReference(r: CtFieldReference<T>) {
            val versionHash = versionHash(r)
            Property(
                    id = Util.uuid(),
                    name = r.simpleName,
                    canonicalName = r.qualifiedName,
                    visibility = when {
                        ModifierKind.PRIVATE in r.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in r.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    },
                    isStatic = ModifierKind.STATIC in r.modifiers,
                    isImmutable = ModifierKind.FINAL in r.modifiers,
                    dataType = r.type.toDataType(),
                    versionHash = versionHash
            ).also { properties[it.canonicalName] = it }
        }

        override fun <T : Any?> visitCtExecutableReference(r: CtExecutableReference<T>) {
            val versionHash = versionHash(r)
            val executable = Executable(
                    id = Util.uuid(),
                    name = r.simpleName,
                    canonicalName = r.canonicalName(),
                    isConstructor = r.isConstructor,
                    dataType = if (r.isConstructor) DataType.VOID else r.type.toDataType(),
                    versionHash = versionHash
            ).also { executables[it.canonicalName] = it }

            r.safeDeclaration().ifPresent { declaration ->
                if (declaration is CtTypeMember) {
                    executable.run {
                        isAbstract = ModifierKind.ABSTRACT in declaration.modifiers || thisType.isInterface
                        visibility = when {
                            ModifierKind.PRIVATE in declaration.modifiers -> Visibility.PRIVATE
                            ModifierKind.PROTECTED in declaration.modifiers -> Visibility.INHERITANCE
                            else -> Visibility.PUBLIC
                        }
                        isStatic = ModifierKind.STATIC in declaration.modifiers
                    }

                    currentExecutable = executable

                    if (declaration is CtMethod<*> && declaration.formalCtTypeParameters.isNotEmpty()) {
                        scan(declaration.formalCtTypeParameters)
                    }

                    scan(declaration.parameters)
                }
            }
        }

        override fun <T : Any?> visitCtParameter(p: CtParameter<T>) {
            assert(currentExecutable != null) { "Check visitCtExecutable and whether state is correctly set." }

            val canonicalName = Parameter.canonicalNameOf(
                    executable = currentExecutable!!,
                    index = p.parent.parameters.indexOf(p)
            )
            Parameter(
                    id = Util.uuid(),
                    name = p.simpleName,
                    canonicalName = canonicalName,
                    dataType = p.type.toDataType()
            ).also { parameters[it.canonicalName] = it }
        }
    }
}