package at.jku.isse.gradient.dal.grpc

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.bytes
import at.jku.isse.gradient.dal.ProjectDao
import at.jku.isse.gradient.dal.ServerManager
import at.jku.isse.gradient.message.Code
import at.jku.isse.gradient.message.Common
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.profiledDebug
import at.jku.isse.gradient.profiledTrace
import com.google.inject.Inject
import com.google.protobuf.Message
import mu.KotlinLogging

private val logger = KotlinLogging.logger {}


class GrpcProjectDao
@Inject constructor(serverManager: ServerManager,
                    private val grpcSerde: GrpcSerde) : ProjectDao {

    private val codeService = serverManager.codeServiceSync()

    override fun storeProject(project: Project) {
        logger.profiledTrace { "Storing project: ${project.canonicalName}" }

        if (project.version.revisions.isEmpty()) {
            logger.debug { "Ignoring revision as it does not contain any elements: ${project.canonicalName}@${project.version.id}" }
        } else {
            val messages = grpcSerde.mapRecursive(project)
            val report = setupReport(project, messages)
            codeService.report(report)
        }

        logger.profiledDebug(ProfilerState.STOP) { "Finished storing project:${project.canonicalName}" }
    }

    private fun setupReport(project: Project, messages: List<Message>): Code.CodeReport {
        val report = Code.CodeReport.newBuilder()
                .setProjectContext(Common.ProjectContext.newBuilder()
                        .setProjectName(project.canonicalName)
                        .setVersionId(Common.UUID.newBuilder()
                                .setBytes(project.version.id.bytes())
                                .build()))
        messages.forEach {
            when (it) {
                is Code.Project -> report.addProjects(it)
                is Code.Version -> report.addVersions(it)
                is Code.Type -> report.addTypes(it)
                is Code.Property -> report.addProperties(it)
                is Code.Executable -> report.addExecutables(it)
                is Code.Parameter -> report.addParameters(it)
                is Code.TypeParameterMapping -> report.addTypeParameterMappings(it)
                is Code.ElementType -> report.addElementTypes(it)
                is Code.Access -> report.addAccesses(it)
                is Code.Invocation -> report.addInvocations(it)
                is Code.Overriding -> report.addOverriding(it)
            }
        }
        return report.build()
    }
}