package at.jku.isse.gradient.lang.java

import at.jku.isse.gradient.model.GradientType
import com.google.common.util.concurrent.AtomicDoubleArray
import spoon.reflect.code.CtFieldAccess
import spoon.reflect.code.CtInvocation
import spoon.reflect.declaration.*
import spoon.reflect.reference.*
import java.io.File
import java.net.URL
import java.nio.file.Path
import java.util.*
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicIntegerArray
import java.util.concurrent.atomic.AtomicLongArray
import java.util.logging.Level
import java.util.regex.Pattern

internal fun CtTypeReference<*>.canonicalName(): String {
    return when (this) {
        is CtArrayTypeReference<*> -> this.componentType.canonicalName()
        is CtTypeParameterReference -> canonicalName()
        else -> box().qualifiedName
    }
}

internal fun CtExecutableReference<*>.canonicalName(): String {
    val declaringType = this.declaringType.canonicalName()
    val executableName = if (this.isConstructor) this.declaringType.simpleName else this.simpleName

    val parameters = this.parameters.joinToString(", ") { parameter ->
        if (parameter.isGenerics && parameter is CtTypeParameterReference) {
            parameter.canonicalName()
        } else {
            parameter.box().qualifiedName
        }
    }

    return "$declaringType#$executableName($parameters)"
}

internal fun CtTypeParameterReference.canonicalName(): String {
    val boundingType = if (this.boundingType == null) "" else ": ${this.boundingType.canonicalName()}"

    return "${this.qualifiedName} $boundingType"
}

internal fun CtElement.canonicalName(): String {
    return when (this) {
        is CtTypeReference<*> -> this.canonicalName()
        is CtExecutableReference<*> -> this.canonicalName()
        is CtTypeParameterReference -> this.canonicalName()
        is CtParameter<*> -> "${this.parent.reference.canonicalName()} <- ${this.simpleName}"
        is CtFieldAccess<*> -> "${this.target} -> ${this.variable}"
        is CtInvocation<*> -> "${this.target} -> ${this.executable}"
        else -> this.toString()
    }
}

abstract class BaseTransformer(private val declarationCache: DeclarationCache) {

    companion object {
        internal fun CtTypeReference<*>.toGradientType(): GradientType {
            val typeFactory = this.factory.Type()

            return when {
                this.isGenerics && !this.isClass && !this.isInterface -> GradientType.UNKNOWN
                this is CtTypeParameterReference && this.boundingType == null -> GradientType.UNKNOWN
                this is CtTypeParameterReference && this.boundingType != null -> this.boundingType.toGradientType()
                this.box().isSubtypeOf(typeFactory.VOID) -> GradientType.VOID
                this.box().isSubtypeOf(typeFactory.BOOLEAN) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.BYTE) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.SHORT) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.INTEGER) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.LONG) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.FLOAT) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.DOUBLE) -> GradientType.NUMBER
                this is CtArrayTypeReference<*> -> this.componentType.box().toGradientType()
                this.isSubtypeOf(typeFactory.get<Number>(Number::class.java).reference.box()) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.get<AtomicBoolean>(AtomicBoolean::class.java).reference) -> GradientType.NUMBER
                this.isSubtypeOf(typeFactory.get<AtomicIntegerArray>(AtomicIntegerArray::class.java).reference) -> GradientType.NUMBER
                this.isSubtypeOf(typeFactory.get<AtomicLongArray>(AtomicLongArray::class.java).reference) -> GradientType.NUMBER
                this.isSubtypeOf(typeFactory.get<AtomicDoubleArray>(AtomicDoubleArray::class.java).reference) -> GradientType.NUMBER
                this.isSubtypeOf(typeFactory.get<Date>(Date::class.java).reference) -> GradientType.NUMBER
                this.box().isSubtypeOf(typeFactory.CHARACTER) -> GradientType.TEXT
                this.box().isSubtypeOf(typeFactory.STRING) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<CharSequence>(CharSequence::class.java).reference.box()) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<File>(File::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Path>(Path::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<URL>(URL::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Pattern>(Pattern::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Level>(Level::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Class<*>>(Class::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Enum<*>>(Enum::class.java).reference) -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.get<Map.Entry<*, *>>(Map.Entry::class.java).reference) && this.actualTypeArguments.size > 1 -> GradientType.TEXT
                this.isSubtypeOf(typeFactory.MAP) && this.actualTypeArguments.size > 1 -> this.actualTypeArguments[1].toGradientType()
                this.isSubtypeOf(typeFactory.ITERABLE) && this.actualTypeArguments.isNotEmpty() -> this.actualTypeArguments.first().toGradientType()
                else -> GradientType.UNKNOWN
            }
        }
    }

    internal fun CtTypeReference<*>.safeDeclaration(): Optional<CtType<*>> {
        return declarationCache.type.get(this)
    }

    internal fun CtFieldReference<*>.safeDeclaration(): Optional<CtField<*>> {
        return declarationCache.property.get(this)
    }

    internal fun CtExecutableReference<*>.safeDeclaration(): Optional<CtExecutable<*>> {
        return declarationCache.executable.get(this)
    }

    internal fun CtParameterReference<*>.safeDeclaration(): Optional<CtParameter<*>> {
        return declarationCache.parameter.get(this)
    }
}