package at.jku.isse.gradient.dal.grpc

import at.jku.isse.gradient.bytes
import at.jku.isse.gradient.message.Code
import at.jku.isse.gradient.message.Common
import at.jku.isse.gradient.model.*
import com.google.protobuf.ByteString
import com.google.protobuf.Message
import mu.KotlinLogging
import java.util.*

private val logger = KotlinLogging.logger {}

private fun toGrpcUUID(uuid: UUID): Common.UUID {
    return Common.UUID.newBuilder().setBytes(uuid.bytes()).build()
}

class GrpcSerde {

    fun mapRecursive(project: Project): List<Message> {
        val nodeMapper = NodeMapper()
        nodeMapper.scan(project)
        val relationshipMapper = RelationshipMapper(nodeMapper.documents)
        relationshipMapper.scan(project)

        return relationshipMapper.documents.values.map(Message.Builder::build)
    }

    fun map(project: Project): Code.Project.Builder {
        return Code.Project.newBuilder()
                .setId(toGrpcUUID(project.id))
                .setName(project.name)
                .setCanonicalName(project.canonicalName)
                .addAllTags(project.tags)
    }

    fun map(version: Version): Code.Version.Builder {
        return Code.Version.newBuilder()
                .setId(toGrpcUUID(version.id))
                .setVersionHash(ByteString.copyFrom(version.versionHash))
                .setTimestamp(version.timestamp)
    }

    fun map(type: Type): Code.Type.Builder {
        return Code.Type.newBuilder()
                .setId(toGrpcUUID(type.id))
                .setName(type.name)
                .setCanonicalName(type.canonicalName)
                .setIsGenerics(type.isGenerics)
                .setIsGradientModel(type.isGradientModel)
                .setAvailabilityValue(type.availability.ordinal)
                .setImplementationQualityValue(type.quality.ordinal)
                .setVisibilityValue(type.visibility.ordinal)
                .setVersionHash(ByteString.copyFrom(type.versionHash.asBytes()))
                .addAllSuperTypes(type.isExtending.map { toGrpcUUID(it.id) })
                .addAllParameterizedBy(type.typeParameters.map { toGrpcUUID(it.id) })
    }

    fun map(property: Property, declaringTypeId: UUID? = null): Code.Property.Builder {
        val propertyBuilder = Code.Property.newBuilder()
                .setId(toGrpcUUID(property.id))
                .setName(property.name)
                .setCanonicalName(property.canonicalName)
                .setIsStatic(property.isStatic)
                .setIsImmutable(property.isImmutable)
                .setGradientTypeValue(property.gradientType.ordinal)
                .setVisibilityValue(property.visibility.ordinal)
                .setVersionHash(ByteString.copyFrom(property.versionHash.asBytes()))
                .addAllShadows(property.shadows.map { toGrpcUUID(it.id) })

        declaringTypeId?.let { propertyBuilder.setDeclaringElement(toGrpcUUID(it)) }
        property.type?.let { propertyBuilder.setType(toGrpcUUID(it.id)) }

        return propertyBuilder
    }

    fun map(executable: Executable, declaringTypeId: UUID? = null): Code.Executable.Builder {
        val executableBuilder = Code.Executable.newBuilder()
                .setId(toGrpcUUID(executable.id))
                .setName(executable.name)
                .setCanonicalName(executable.canonicalName)
                .setIsConstructor(executable.isConstructor)
                .setIsAbstract(executable.isAbstract)
                .setIsStatic(executable.isStatic)
                .setVisibilityValue(executable.visibility.ordinal)
                .setAccessorTypeValue(executable.accessor.ordinal)
                .setGradientTypeValue(executable.gradientType.ordinal)
                .setVersionHash(ByteString.copyFrom(executable.versionHash.asBytes()))
                .addAllInvokes(executable.invokes.map { toGrpcUUID(it.id) })
                .addAllAccesses(executable.accesses.map { toGrpcUUID(it.id) })
                .addAllOverrides(executable.overrides.map { toGrpcUUID(it.id) })
                .addAllReturns(executable.returns.map { toGrpcUUID(it.id) })
                .addAllParameterizedBy(executable.typeParameters.map { toGrpcUUID(it.id) })

        declaringTypeId?.let { executableBuilder.setDeclaringElement(toGrpcUUID(it)) }
        executable.type?.let { executableBuilder.setType(toGrpcUUID(it.id)) }

        return executableBuilder
    }

    fun map(parameter: Parameter, declaringExecutableId: UUID? = null): Code.Parameter.Builder {
        val parameterBuilder = Code.Parameter.newBuilder()
                .setId(toGrpcUUID(parameter.id))
                .setName(parameter.name)
                .setCanonicalName(parameter.canonicalName)
                .setGradientTypeValue(parameter.gradientType.ordinal)

        declaringExecutableId?.let { parameterBuilder.setDeclaringElement(toGrpcUUID(it)) }
        parameter.type?.let { parameterBuilder.setType(toGrpcUUID(it.id)) }

        return parameterBuilder
    }

    fun map(typeParameterMapping: TypeParameterMapping, declaringElement: UUID? = null): Code.TypeParameterMapping.Builder {
        val builder = Code.TypeParameterMapping.newBuilder()
                .setId(toGrpcUUID(typeParameterMapping.id))
                .setCanonicalName(typeParameterMapping.canonicalName)
                .setActualType(toGrpcUUID(typeParameterMapping.actualType.id))
                .setParameter(toGrpcUUID(typeParameterMapping.parameter.id))

        declaringElement?.let { builder.setDeclaringElement(toGrpcUUID(it)) }
        return builder
    }

    fun map(elementType: ElementType): Code.ElementType.Builder {
        return Code.ElementType.newBuilder()
                .setId(toGrpcUUID(elementType.id))
                .setCanonicalName(elementType.canonicalName)
                .setCardinalityValue(elementType.cardinality.ordinal)
                .setTypeOf(toGrpcUUID(elementType.typeOf.id))
                .addAllMappings(elementType.typeParameterMappings.map { toGrpcUUID(it.id) })
    }

    fun map(access: Access): Code.Access.Builder {
        val builder = Code.Access.newBuilder()
                .setId(toGrpcUUID(access.id))
                .setCanonicalName(access.canonicalName)
                .setSource(toGrpcUUID(access.source.id))
                .setTarget(toGrpcUUID(access.target.id))
                .setCardinalityValue(access.cardinality.ordinal)
                .setAccessTypeValue(access.accessType.ordinal)

        access.delegate?.let { builder.setVia(toGrpcUUID(access.delegate.id)) }

        return builder
    }

    fun map(invocation: Invocation): Code.Invocation.Builder {
        val builder = Code.Invocation.newBuilder()
                .setId(toGrpcUUID(invocation.id))
                .setCanonicalName(invocation.canonicalName)
                .setSource(toGrpcUUID(invocation.source.id))
                .setTarget(toGrpcUUID(invocation.target.id))
                .setCardinalityValue(invocation.cardinality.ordinal)

        invocation.delegate?.let { builder.setVia(toGrpcUUID(it.id)) }

        return builder
    }

    fun map(overriding: Overriding): Code.Overriding.Builder {
        return Code.Overriding.newBuilder()
                .setId(toGrpcUUID(overriding.id))
                .setCanonicalName(overriding.canonicalName)
                .setSource(toGrpcUUID(overriding.source.id))
                .setTarget(toGrpcUUID(overriding.target.id))
                .setOverridingQualityValue(overriding.overridingQuality.ordinal)
    }

    private inner class RelationshipMapper(val documents: MutableMap<UUID, Message.Builder>) : ModelVisitor {
        val visitedElements = mutableSetOf<UUID>()

        override fun scan(entity: ModelVisitable) {
            if (entity is Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        override fun visitProject(e: Project) = scan(e.version)

        override fun visitVersion(e: Version) = scan(e.revisions)

        override fun visitType(e: Type) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            val typeId = toGrpcUUID(e.id)
            e.properties.map { documents[it.id] }
                    .filterIsInstance<Code.Property.Builder>()
                    .forEach { it.declaringElement = typeId }
            e.executables.map { documents[it.id] }
                    .filterIsInstance<Code.Executable.Builder>()
                    .forEach { it.declaringElement = typeId }

            scan(e.properties)
            scan(e.inheritedProperties)
            scan(e.executables)
            scan(e.inheritedExecutables)
            scan(e.typeParameters)
            scan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            scan(e.shadows)
            e.type?.let { scan(it) }
        }

        override fun visitExecutable(e: Executable) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            val typeId = toGrpcUUID(e.id)
            e.parameters.map { documents[it.id] }
                    .filterIsInstance<Code.Parameter.Builder>()
                    .forEach { it.declaringElement = typeId }

            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            e.type?.let { scan(it) }
            scan(e.invokes)
            scan(e.accesses)
            scan(e.typeParameters)
        }

        override fun visitParameter(e: Parameter) {
            e.type?.let { scan(it) }
        }

        override fun visitElementType(e: ElementType) {
            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            scan(e.actualType)
            scan(e.parameter)
        }

        override fun visitAccess(e: Access) {
            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitInvocation(e: Invocation) {
            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitOverriding(e: Overriding) {
            scan(e.source)
            scan(e.target)
        }
    }

    private inner class NodeMapper : ModelVisitor {
        val visitedElements = mutableSetOf<UUID>()
        val documents: MutableMap<UUID, Message.Builder> = mutableMapOf()

        override fun scan(entity: ModelVisitable) {
            if (entity is Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        override fun visitProject(e: Project) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents[e.id] = map(e)

            scan(e.version)
        }

        override fun visitVersion(e: Version) {
            documents[e.id] = map(e)

            scan(e.revisions)
        }

        override fun visitType(e: Type) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents[e.id] = map(e)

            scan(e.properties)
            scan(e.inheritedProperties)
            scan(e.executables)
            scan(e.inheritedExecutables)
            scan(e.typeParameters)
            scan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            documents[e.id] = map(e)

            scan(e.shadows)
            e.type?.let { scan(it) }
        }

        override fun visitExecutable(e: Executable) {
            documents[e.id] = map(e)

            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            e.type?.let { scan(it) }
            scan(e.invokes)
            scan(e.accesses)
            scan(e.typeParameters)
        }

        override fun visitParameter(e: Parameter) {
            documents[e.id] = map(e)

            e.type?.let { scan(it) }
        }

        override fun visitElementType(e: ElementType) {
            documents[e.id] = map(e)

            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            documents[e.id] = map(e)

            scan(e.actualType)
            scan(e.parameter)
        }

        override fun visitAccess(e: Access) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitInvocation(e: Invocation) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitOverriding(e: Overriding) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
        }
    }
}