package at.jku.isse.gradient.dal

import at.jku.isse.gradient.GradientConfig
import at.jku.isse.gradient.GradientEvents
import at.jku.isse.gradient.message.CodeServiceGrpc
import at.jku.isse.gradient.message.EventServiceGrpc
import com.google.common.eventbus.EventBus
import com.google.common.eventbus.Subscribe
import com.google.inject.Inject
import io.grpc.ConnectivityState
import io.grpc.ManagedChannel
import io.grpc.ManagedChannelBuilder
import mu.KotlinLogging
import java.util.concurrent.TimeUnit

private val logger = KotlinLogging.logger {}


class ServerManager
@Inject constructor(config: GradientConfig,
                    gradientEventBus: EventBus) {

    private val grpcChannel: ManagedChannel by lazy {
        ManagedChannelBuilder
                .forTarget(config.gradientServerConnectionString())
                .usePlaintext()
                .build()
    }

    init {
        gradientEventBus.register(this)
    }

    fun eventService(): EventServiceGrpc.EventServiceStub {
        return EventServiceGrpc.newStub(grpcChannel)
    }

    fun codeServiceSync(): CodeServiceGrpc.CodeServiceBlockingStub {
        return CodeServiceGrpc.newBlockingStub(grpcChannel)
    }

    @Synchronized
    fun backendAvailable(): Boolean {
        return ConnectivityState.SHUTDOWN != grpcChannel.getState(true)
    }

    @Subscribe
    @Suppress("UNUSED_PARAMETER")
    fun shutdown(shutdown: GradientEvents.Shutdown) {
        logger.debug { "Closing the database clients." }
        grpcChannel.let {
            it.shutdown()
            it.awaitTermination(10, TimeUnit.SECONDS)
            it.shutdownNow()
        }
    }
}