package at.jku.isse.gradient

import at.jku.isse.gradient.message.EventServiceGrpc
import com.fasterxml.uuid.Generators
import com.google.common.base.Stopwatch
import com.google.protobuf.ByteString
import io.grpc.ConnectivityState
import io.grpc.ManagedChannel
import mu.KLogger
import java.nio.ByteBuffer
import java.util.*

enum class ProfilerState {
    START, RESET, STOP
}

private val stopwatches = object : ThreadLocal<MutableMap<KLogger, Stopwatch>>() {
    override fun initialValue(): MutableMap<KLogger, Stopwatch> {
        return mutableMapOf()
    }
}

fun KLogger.profiledInfo(state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    log(this, this::info, state, msg)
}

fun KLogger.profiledDebug(state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    log(this, this::debug, state, msg)
}

fun KLogger.profiledTrace(state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    log(this, this::trace, state, msg)
}

private fun log(logger: KLogger, logMethod: (() -> Any?) -> Unit, state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    when (state) {
        ProfilerState.START -> {
            stopwatches.get()[logger] = Stopwatch.createStarted()
            logMethod(msg)
        }
        ProfilerState.RESET -> {
            val stopwatch = stopwatches.get().getOrPut(logger) {
                Stopwatch.createStarted()
            }

            logMethod { "${msg.invoke()} [Finished after $stopwatch]" }
            stopwatch.reset().start()
        }
        ProfilerState.STOP -> {
            stopwatches.get().remove(logger)?.let {
                logMethod { "${msg.invoke()} [Finished after ${it.stop()}]" }
            }
        }

    }
}

object Util {
    private val generator = Generators.timeBasedGenerator()
    fun uuid(): UUID {
        return generator.generate()
    }
}

fun UUID.bytes(): ByteString {
    val array = ByteBuffer.wrap(ByteArray(16)).putLong(this.mostSignificantBits).putLong(this.leastSignificantBits).array()
    return ByteString.copyFrom(array)
}

fun EventServiceGrpc.EventServiceBlockingStub.serverAvailable(): Boolean {
    val ch = channel
    return if (ch is ManagedChannel) {
        ch.getState(true) == ConnectivityState.READY
    } else {
        false
    }
}