package at.jku.isse.gradient.service.spoon

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.Util
import at.jku.isse.gradient.dal.ProjectDao
import at.jku.isse.gradient.lang.java.ModelTransformer
import at.jku.isse.gradient.lang.java.StructuralModel
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.model.StructuralCache
import at.jku.isse.gradient.model.Version
import at.jku.isse.gradient.model.Versionable
import at.jku.isse.gradient.profiledDebug
import at.jku.isse.gradient.profiledInfo
import at.jku.isse.gradient.service.StructuralService
import com.google.common.hash.HashCode
import com.google.common.hash.Hashing
import com.google.inject.Inject
import mu.KotlinLogging
import spoon.Launcher
import spoon.reflect.factory.Factory
import spoon.support.compiler.FileSystemFile
import spoon.support.compiler.FileSystemFolder
import spoon.support.compiler.ZipFolder
import java.nio.file.Files
import java.nio.file.Paths
import java.util.*

private val logger = KotlinLogging.logger {}


class StructuralService
@Inject internal constructor(private val projectDao: ProjectDao,
                             private val modelTransformer: ModelTransformer) : StructuralService {

    companion object {
        private val emptyHash = HashCode.fromInt(0)
    }

    override fun analyze(groupName: String,
                         projectName: String,
                         sourcePaths: Collection<String>,
                         classPaths: Collection<String>,
                         gradientModelRegex: Collection<String>): StructuralModel {
        require(groupName.isNotBlank())
        require(projectName.isNotBlank())
        require(sourcePaths.isNotEmpty())

        val canonicalProjectName = "$groupName.$projectName"

        logger.profiledDebug { "Building model: $canonicalProjectName" }

        val modelComponents = buildModel(sourcePaths, classPaths, gradientModelRegex)

        logger.profiledInfo(ProfilerState.RESET) { "Finished building the model: ${modelComponents.types.size} types" }

        val version = createVersion(modelComponents)
        val project = Project(Util.uuid(), projectName, canonicalProjectName, version)

        modelComponents.project = project
        modelComponents.version = version
        logger.profiledDebug(ProfilerState.RESET) { "Persisting the model" }

        projectDao.storeProject(project)

        logger.profiledInfo(ProfilerState.STOP) { "Finished persisting the model" }

        return modelComponents
    }

    override fun prepareStructuralCache(structuralModel: StructuralModel): StructuralCache {
        return StructuralCache(
                structuralModel.project.canonicalName,
                structuralModel.version.id
        )
    }


    private fun buildModel(inputPaths: Collection<String>,
                           classPaths: Collection<String>,
                           additionalGradientModelRegex: Collection<String>): StructuralModel {
        val spoonModel = buildSpoonModel(inputPaths, classPaths)
        return modelTransformer.transform(spoonModel, ModelTransformer.TypeUniverse.DECLARED, additionalGradientModelRegex)
    }

    private fun createVersion(structuralModel: StructuralModel): Version {
        val typeHash = structuralModel.types.keys
                .sorted()
                .map { structuralModel.types[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val propertyHash = structuralModel.properties.keys
                .sorted()
                .map { structuralModel.properties[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val executableHash = structuralModel.executables.keys
                .sorted()
                .map { structuralModel.executables[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val versionHash = listOf(typeHash, propertyHash, executableHash)
                .filter { it != emptyHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val versionables = structuralModel
                .filterIsInstance(Versionable::class.java)
                .toMutableSet()

        // Use strong uuid for versions
        return Version(UUID.randomUUID(), versionHash.asBytes(), versionables)
    }

    private fun buildSpoonModel(inputPaths: Collection<String>, classPaths: Collection<String>): Factory {
        return Launcher().let { launcher ->
            inputPaths
                    .distinct()
                    .map { Paths.get(it) }
                    .filter { Files.exists(it) }
                    .forEach {
                        val resource = when {
                            Files.isDirectory(it) -> FileSystemFolder(it.toFile())
                            it.toString().endsWith(".jar") -> ZipFolder(it.toFile())
                            else -> FileSystemFile(it.toFile())
                        }

                        launcher.addInputResource(resource)
                    }

            if (classPaths.isNotEmpty()) launcher.environment.sourceClasspath = classPaths.toTypedArray()
            launcher.environment.noClasspath = classPaths.isEmpty()
            launcher.buildModel()
            launcher.factory
        }
    }
}