package at.jku.isse.gradient.service.grpc

import at.jku.isse.gradient.GradientConfig
import at.jku.isse.gradient.GradientEvents
import at.jku.isse.gradient.Util
import at.jku.isse.gradient.bytes
import at.jku.isse.gradient.message.Context
import at.jku.isse.gradient.message.Event
import at.jku.isse.gradient.message.EventBatch
import at.jku.isse.gradient.message.EventServiceGrpc
import at.jku.isse.gradient.model.EventType
import at.jku.isse.gradient.model.GradientType
import at.jku.isse.gradient.model.StructuralCache
import at.jku.isse.gradient.runtime.__Gradient_Observable__
import at.jku.isse.gradient.service.EventService
import com.google.common.eventbus.EventBus
import com.google.common.eventbus.Subscribe
import com.google.common.util.concurrent.AtomicDoubleArray
import com.google.inject.Inject
import com.google.inject.Provider
import com.google.inject.name.Named
import mu.KotlinLogging
import java.io.File
import java.net.URL
import java.nio.file.Path
import java.time.Instant
import java.util.*
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicIntegerArray
import java.util.concurrent.atomic.AtomicLongArray
import java.util.logging.Level
import java.util.regex.Pattern
import at.jku.isse.gradient.message.EventType as GrpcEventType
import at.jku.isse.gradient.message.GradientType as GrpcGradientType

private val logger = KotlinLogging.logger {}

private data class Frame(val id: UUID,
                         val name: String)

private data class FrameStack(val threadId: Long,
                              val processId: UUID = Util.uuid(),
                              val frames: Stack<Frame> = Stack(),
                              var frameDepth: Int = 0) {
    init {
        frames.push(Frame(UUID.randomUUID(), "<alien>"))
        frameDepth = 1
    }
}

/**
 * State-full but thread-safe.
 */
class GrpcEventService
@Inject internal constructor(@Named("runtimeId") private val runtimeId: UUID,
                             structuralCacheProvider: Provider<StructuralCache?>,
                             gradientConfig: GradientConfig,
                             eventBus: EventBus,
                             private val eventServiceStub: EventServiceGrpc.EventServiceBlockingStub) : EventService {

    private val iterableUnwindDepth = gradientConfig.iterableUnwindDepth()
    private val projectName: String?
    private val versionId: UUID?
    private val stacks: MutableMap<Long, FrameStack> = mutableMapOf()

    private val eventBuffer: Vector<Event> = Vector(gradientConfig.observationCacheSize())
    private val bufferSize: Int = gradientConfig.observationCacheSize()

    init {
        val structuralCache = structuralCacheProvider.get()
        if (structuralCache != null) {
            projectName = structuralCache.projectName
            versionId = structuralCache.versionId

            eventBus.register(this)
            logger.debug { "Reporting to $projectName @ $versionId" }
        } else {
            projectName = null
            versionId = null

            logger.warn { "Structural cache is not available, no events will be reported." }
        }
    }

    @Subscribe
    fun cleanupSignal(@Suppress("UNUSED_PARAMETER") cleanup: GradientEvents.Cleanup) {
        flush()
    }

    private fun getFrameStack(): FrameStack {
        val threadId = Thread.currentThread().id
        val frameStack: FrameStack

        if (threadId !in stacks) {
            frameStack = FrameStack(threadId)
            stacks[threadId] = frameStack
        } else {
            frameStack = stacks[threadId]!!
        }

        return frameStack
    }

    override fun reportPropertyRead(elementName: String, objectId: UUID, obj: Any?) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        val frame = frameStack.frames.peek()
        val events = toValue(obj).map { (valueType, value) ->
            newEvent(EventType.READ, frameStack.processId, objectId, Util.uuid(), elementName, frame.id, frame.name, valueType, value)
        }
        report(events)
    }

    override fun reportPropertyWrite(elementName: String, objectId: UUID, obj: Any?) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        val frame = frameStack.frames.peek()
        val events = toValue(obj).map { (valueType, value) ->
            newEvent(EventType.WRITE, frameStack.processId, objectId, Util.uuid(), elementName, frame.id, frame.name, valueType, value)
        }
        report(events)
    }

    override fun reportExecutableCall(elementName: String, objectId: UUID) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        val eventId = Util.uuid()
        val frame = frameStack.frames.peek()
        val event = newEvent(EventType.CALL, frameStack.processId, objectId, eventId, elementName, frame.id, frame.name)

        frameStack.frameDepth++
        frameStack.frames.push(Frame(eventId, elementName))

        report(event)
    }

    override fun reportExecutableParameter(elementName: String, objectId: UUID, obj: Any?) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        val frame = frameStack.frames.peek()
        val events = toValue(obj).map { (valueType, value) ->
            newEvent(EventType.RECEIVE, frameStack.processId, objectId, Util.uuid(), elementName, frame.id, frame.name, valueType, value)
        }
        report(events)
    }

    override fun reportExecutableReturn(elementName: String, objectId: UUID, obj: Any?) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        frameStack.frameDepth--
        val frame = frameStack.frames.pop()
        if (frameStack.frameDepth == 0) {
            assert(frameStack.frames.empty())
            stacks.remove(frameStack.threadId)
        }

        val events = toValue(obj).map { (valueType, value) ->
            newEvent(EventType.RETURN, frameStack.processId, objectId, Util.uuid(), elementName, frame.id, frame.name, valueType, value)
        }
        report(events)
    }

    override fun reportExecutableException(elementName: String, objectId: UUID, exception: Throwable) {
        if (projectName == null || versionId == null) return
        assert(elementName.isNotBlank())

        val frameStack = getFrameStack()
        assert(frameStack.frameDepth > 0)

        frameStack.frameDepth--
        val frame = frameStack.frames.pop()
        if (frameStack.frameDepth == 0) {
            assert(frameStack.frames.empty())
            stacks.remove(frameStack.threadId)
        }

        val event = newEvent(EventType.EXCEPT, frameStack.processId, objectId, Util.uuid(), elementName, frame.id, frame.name, GradientType.TEXT, exception::class.qualifiedName)
        report(event)
    }


    private fun newEvent(eventType: EventType, processId: UUID, objectId: UUID, eventId: UUID, elementName: String, frameId: UUID, frameName: String,
                         valueType: GradientType = GradientType.VOID, value: Any? = null): Event {
        assert(value != null || valueType == GradientType.VOID) { "The value is null while still having a gradient type that is not void." }

        val context = Context.newBuilder()
                .setRuntimeId(runtimeId.bytes())
                .setProcessId(processId.bytes())
                .setObjectId(objectId.bytes())
                .setFrameId(frameId.bytes())
                .setFrameName(frameName)
                .setTime(Instant.now().toEpochMilli())
                .build()

        val eventBuilder = Event.newBuilder()
                .setEventType(GrpcEventType.forNumber(eventType.value))
                .setId(eventId.bytes())
                .setElementName(elementName)
                .setContext(context)
                .setValueType(GrpcGradientType.forNumber(valueType.value))


        if (value != null) {
            assert(value is String || value is Double || value is UUID) { "Value did not have the proper type: ${value::class}" }
            assert(value != GradientType.NUMBER || value is Double) { "Number value is not a double: $elementName" }
            when (valueType) {
                GradientType.TEXT -> eventBuilder.textValue = value as String
                GradientType.NUMBER -> eventBuilder.numberValue = value as Double
                GradientType.REFERENCE -> eventBuilder.referenceValue = (value as UUID).bytes()
                GradientType.UNKNOWN -> eventBuilder.textValue = value as String
                GradientType.VOID -> {
                }
            }
        }
        return eventBuilder.build()
    }

    private fun report(event: Event) {
        if (projectName == null || versionId == null) return

        eventBuffer.add(event)

        if (eventBuffer.size >= bufferSize) flush()
    }

    private fun report(events: List<Event>) {
        if (projectName == null || versionId == null || events.isEmpty()) return

        eventBuffer.addAll(events)

        if (eventBuffer.size >= bufferSize) flush()
    }

    private fun flush() {
        if (projectName == null || versionId == null || eventBuffer.isEmpty()) return

        synchronized(eventBuffer) {
            val eventBatch = EventBatch.newBuilder()
                    .setProjectName(projectName)
                    .setVersionId(versionId.bytes())
                    .addAllEvents(eventBuffer)
                    .build()

            eventServiceStub.reportEvents(eventBatch)
            eventBuffer.clear()
        }
    }

    private fun toValue(obj: Any?, unwindDepth: Int = iterableUnwindDepth): List<Pair<GradientType, Any?>> {
        return when (obj) {
            is __Gradient_Observable__ -> listOf(GradientType.REFERENCE to obj.__gradient_id__())
            is Boolean -> listOf(GradientType.NUMBER to (if (obj) 1 else 0).toDouble())
            is AtomicBoolean -> listOf(GradientType.NUMBER to (if (obj.get()) 1 else 0).toDouble())
            is Number -> listOf(GradientType.NUMBER to obj.toDouble())
            is Date -> listOf(GradientType.NUMBER to obj.time.toDouble())
            is AtomicIntegerArray -> iterableToValues((0..obj.length()).map { obj.get(it) }, unwindDepth)
            is AtomicLongArray -> iterableToValues((0..obj.length()).map { obj.get(it) }, unwindDepth)
            is AtomicDoubleArray -> iterableToValues((0..obj.length()).map { obj.get(it) }, unwindDepth)
            is Char,
            is CharSequence -> listOf(GradientType.TEXT to obj.toString())
            is File -> listOf(GradientType.TEXT to obj.absolutePath)
            is Path -> listOf(GradientType.TEXT to obj.toAbsolutePath().toString())
            is URL -> listOf(GradientType.TEXT to obj.toString())
            is Pattern -> listOf(GradientType.TEXT to obj.pattern())
            is Level -> listOf(GradientType.TEXT to obj.name)
            is Class<*> -> listOf(GradientType.TEXT to obj.canonicalName)
            is Enum<*> -> listOf(GradientType.TEXT to obj.name)
            is CharArray -> listOf(GradientType.TEXT to obj.joinToString(""))
            is ByteArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is ShortArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is IntArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is LongArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is FloatArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is DoubleArray -> iterableToValues(obj.asIterable(), unwindDepth)
            is Array<*> -> iterableToValues(obj.asIterable(), unwindDepth)
            is Map.Entry<*, *> -> toValue(obj.value)
            is Map<*, *> -> iterableToValues(obj.values, unwindDepth)
            is Collection<*> -> iterableToValues(obj.asIterable(), unwindDepth)
            is Iterator<*> -> iterableToValues(Iterable { obj.iterator() }, unwindDepth)
            else -> if (obj == null) listOf(GradientType.VOID to null) else listOf(Pair(GradientType.UNKNOWN, obj::class.qualifiedName))
        }
    }

    private fun iterableToValues(iterable: Iterable<Any?>, unwindDepth: Int): List<Pair<GradientType, Any?>> {
        val result = mutableListOf<Pair<GradientType, Any?>>()
        for (obj in iterable.asIterable()) {
            if (result.size < unwindDepth) {
                val values = toValue(obj, iterableUnwindDepth - result.size)
                result.addAll(values)
            }
        }

        return result
    }
}