package at.jku.isse.gradient.dal.mongo

import at.jku.isse.gradient.model.*
import mu.KotlinLogging
import org.bson.Document
import java.util.*

private val logger = KotlinLogging.logger {}

class QueryMapper {

    fun mapAll(project: Project): List<Document> {
        val nodeMapper = NodeMapper()
        nodeMapper.scan(project)
        val relationshipMapper = RelationshipMapper(nodeMapper.documents)
        relationshipMapper.scan(project)

        return relationshipMapper.documents.values.toList()
    }

    fun map(project: Project): Document {
        return Document("tags", listOf("Project"))
                .append("id", project.id)
                .append("name", project.name)
                .append("canonical_name", project.canonicalName)
    }

    fun map(version: Version): Document {
        return Document("tags", listOf("Version"))
                .append("id", version.id)
                .append("version_hash", version.versionHash)
                .append("timestamp", version.timestamp)
    }

    fun map(type: Type): Document {
        return Document("tags", listOf("Versionable", "Type"))
                .append("id", type.id)
                .append("name", type.name)
                .append("canonical_name", type.canonicalName)
                .append("availability", type.availability.name)
                .append("implementation_quality", type.quality.name)
                .append("visibility", type.visibility.name)
                .append("is_generics", type.isGenerics)
                .append("is_gradient_model", type.isGradientModel)
                .append("super_types", type.isExtending.map { it.id })
                .append("version_hash", type.versionHash.asBytes())
                .append("parameterized_by", type.typeParameters.map { it.id })
    }

    fun map(property: Property, declaringTypeId: UUID? = null): Document {
        val document = Document("tags", listOf("Versionable", "Property"))
                .append("id", property.id)
                .append("name", property.name)
                .append("canonical_name", property.canonicalName)
                .append("visibility", property.visibility.name)
                .append("is_static", property.isStatic)
                .append("is_immutable", property.isImmutable)
                .append("gradient_type", property.gradientType.name)
                .append("version_hash", property.versionHash.asBytes())
                .append("shadows", property.shadows.map { it.id })

        declaringTypeId?.let { document.append("declaring_type", it) }
        property.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(executable: Executable, declaringTypeId: UUID? = null): Document {
        val document = Document("tags", listOf("Versionable", "Executable"))
                .append("id", executable.id)
                .append("name", executable.name)
                .append("canonical_name", executable.canonicalName)
                .append("is_constructor", executable.isConstructor)
                .append("is_abstract", executable.isAbstract)
                .append("is_static", executable.isStatic)
                .append("visibility", executable.visibility.name)
                .append("accessor_type", executable.accessor.name)
                .append("gradient_type", executable.gradientType.name)
                .append("version_hash", executable.versionHash.asBytes())
                .append("invokes", executable.invokes.map { it.id })
                .append("accesses", executable.accesses.map { it.id })
                .append("overrides", executable.overrides.map { it.id })
                .append("returns", executable.returns.map { it.id })
                .append("parametrized_by", executable.typeParameters.map { it.id })

        declaringTypeId?.let { document.append("declaring_type", it) }
        executable.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(parameter: Parameter, declaringExecutableId: UUID? = null): Document {
        val document = Document("tags", listOf("Parameter"))
                .append("id", parameter.id)
                .append("name", parameter.name)
                .append("canonical_name", parameter.canonicalName)
                .append("gradient_type", parameter.gradientType.name)

        declaringExecutableId?.let { document.append("declaring_executable", it) }
        parameter.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(typeParameterMapping: TypeParameterMapping, declaringElement: UUID? = null): Document {
        val document = Document("tags", listOf("TypeParameter"))
                .append("id", typeParameterMapping.id)
                .append("canonical_name", typeParameterMapping.canonicalName)
                .append("actual_type", typeParameterMapping.actualType.id)
                .append("parameter", typeParameterMapping.parameter.id)

        declaringElement?.let { document.append("declaring_element", it) }
        return document
    }

    fun map(elementType: ElementType): Document {
        return Document("tags", listOf("ElementType"))
                .append("id", elementType.id)
                .append("canonical_name", elementType.canonicalName)
                .append("cardinality", elementType.cardinality.name)
                .append("type_of", elementType.typeOf.id)
                .append("mappings", elementType.typeParameterMappings.map { it.id })
    }

    fun map(access: Access): Document {
        val document = Document("tags", listOf("Access"))
                .append("id", access.id)
                .append("canonical_name", access.canonicalName)
                .append("source", access.source.id)
                .append("target", access.target.id)
                .append("cardinality", access.cardinality.name)
                .append("access_type", access.accessType.name)

        access.delegate?.let { document.append("via", access.delegate.id) }

        return document
    }

    fun map(invocation: Invocation): Document {
        val document = Document("tags", listOf("Invocation"))
                .append("id", invocation.id)
                .append("canonical_name", invocation.canonicalName)
                .append("source", invocation.source.id)
                .append("target", invocation.target.id)
                .append("cardinality", invocation.cardinality.name)

        invocation.delegate?.let { document.append("via", it.id) }

        return document
    }

    fun map(overriding: Overriding): Document {
        return Document("tags", listOf("Overriding"))
                .append("id", overriding.id)
                .append("canonical_name", overriding.canonicalName)
                .append("source", overriding.source.id)
                .append("target", overriding.target.id)
                .append("quality", overriding.overridingQuality.name)
    }

    private inner class RelationshipMapper(val documents: MutableMap<UUID, Document>) : ModelVisitor {
        val visitedElements = mutableSetOf<UUID>()

        override fun scan(entity: ModelVisitable) {
            if (entity is Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        override fun visitProject(e: Project) = scan(e.version)

        override fun visitVersion(e: Version) = scan(e.revisions)

        override fun visitType(e: Type) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            e.properties.forEach { documents[it.id]!!["declaring_type"] = e.id }
            e.executables.forEach { documents[it.id]!!["declaring_type"] = e.id }

            scan(e.properties)
            scan(e.inheritedProperties)
            scan(e.executables)
            scan(e.inheritedExecutables)
            scan(e.typeParameters)
            scan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            scan(e.shadows)
            e.type?.let { scan(it) }
        }

        override fun visitExecutable(e: Executable) {
            assert(e.id in documents) { "Expected the element to exist in the documents. Has every node been mapped?" }

            e.parameters.forEach { documents[it.id]!!["declaring_executable"] = e.id }

            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            e.type?.let { scan(it) }
            scan(e.invokes)
            scan(e.accesses)
            scan(e.typeParameters)
        }

        override fun visitParameter(e: Parameter) {
            e.type?.let { scan(it) }
        }

        override fun visitElementType(e: ElementType) {
            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            scan(e.actualType)
            scan(e.parameter)
        }

        override fun visitAccess(e: Access) {
            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitInvocation(e: Invocation) {
            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitOverriding(e: Overriding) {
            scan(e.source)
            scan(e.target)
        }
    }

    private inner class NodeMapper : ModelVisitor {
        val visitedElements = mutableSetOf<UUID>()
        val documents: MutableMap<UUID, Document> = mutableMapOf()

        override fun scan(entity: ModelVisitable) {
            if (entity is Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        override fun visitProject(e: Project) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents[e.id] = map(e)

            scan(e.version)
        }

        override fun visitVersion(e: Version) {
            documents[e.id] = map(e)

            scan(e.revisions)
        }

        override fun visitType(e: Type) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents[e.id] = map(e)

            scan(e.properties)
            scan(e.inheritedProperties)
            scan(e.executables)
            scan(e.inheritedExecutables)
            scan(e.typeParameters)
            scan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            documents[e.id] = map(e)

            scan(e.shadows)
            e.type?.let { scan(it) }
        }

        override fun visitExecutable(e: Executable) {
            documents[e.id] = map(e)

            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            e.type?.let { scan(it) }
            scan(e.invokes)
            scan(e.accesses)
            scan(e.typeParameters)
        }

        override fun visitParameter(e: Parameter) {
            documents[e.id] = map(e)

            e.type?.let { scan(it) }
        }

        override fun visitElementType(e: ElementType) {
            documents[e.id] = map(e)

            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            documents[e.id] = map(e)

            scan(e.actualType)
            scan(e.parameter)
        }

        override fun visitAccess(e: Access) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitInvocation(e: Invocation) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitOverriding(e: Overriding) {
            documents[e.id] = map(e)

            scan(e.source)
            scan(e.target)
        }
    }
}