package at.jku.isse.gradient.dal.mongo

import at.jku.isse.gradient.*
import at.jku.isse.gradient.dal.ProjectDao
import at.jku.isse.gradient.dal.ServerManager
import at.jku.isse.gradient.dal.toMongoCollectionName
import at.jku.isse.gradient.dal.toMongoDatabaseName
import at.jku.isse.gradient.message.StructuralReport
import at.jku.isse.gradient.message.StructuralServiceGrpc
import at.jku.isse.gradient.model.Project
import com.google.inject.Inject
import mu.KotlinLogging
import org.bson.Document
import java.util.*

private val logger = KotlinLogging.logger {}

private fun <T> Document.getList(key: String): List<T> {
    @Suppress("UNCHECKED_CAST")
    return get(key) as List<T>
}

class MongoProjectDao
@Inject constructor(private val serverManager: ServerManager,
                    private val queryMapper: QueryMapper,
                    private val structuralServiceGrpc: StructuralServiceGrpc.StructuralServiceBlockingStub) : ProjectDao {

    override fun storeProject(project: Project) {
        logger.profiledTrace { "Storing project: ${project.canonicalName}" }

        if (project.version.revisions.isEmpty()) {
            logger.debug { "Ignoring revision as it does not contain any elements: ${project.canonicalName}@${project.version.id}" }
        } else {
            // FIXME let the gradient server acknowledge the registration by returning the collection names.
            reportProjectToGradientServer(project, project.version.id)

            val similarVersionId = versionExistsNot(project)
            if (similarVersionId != null) {
                logger.debug { "Found version with similar versionHash ($similarVersionId). Ignoring currently processed version." }
            } else {
                storeProjectCollection(project)
                storeStructuralCollection(project)
            }
        }

        logger.profiledDebug(ProfilerState.STOP) { "Finished storing project:${project.canonicalName}" }
    }

    private fun storeProjectCollection(project: Project) {
        serverManager.mongoDatabase(project.canonicalName)?.let { db ->
            if (MongoDBConventions.PROJECT_COLLECTION in db.listCollectionNames()) {
                val collection = db.getCollection(MongoDBConventions.PROJECT_COLLECTION)

                val projectQuery = Document("tags", "Project")
                        .append("canonicalName", project.canonicalName)

                val result = collection.find(projectQuery).first()
                if (result == null) {
                    collection.insertOne(queryMapper.map(project))
                }

                collection.insertOne(queryMapper.map(project.version))
            }
        }
    }

    private fun storeStructuralCollection(project: Project) {
        serverManager.mongoDatabase(project.canonicalName)?.let { db ->
            val collectionName = toMongoCollectionName(project.version.id.toString(), MongoDBConventions.STRUCTURAL_COLLECTION_POSTFIX)
            if (collectionName in db.listCollectionNames()) {
                val collection = db.getCollection(collectionName)

                assert(collection.find().first() == null) { "Expecting to hav created a new version." }
                val documents = queryMapper.mapAll(project)

                assert("Project" in documents[0].getList<String>("tags")) { "Expected the project to be the first document." }
                assert("Version" in documents[1].getList<String>("tags")) { "Expected the version to be the first document." }
                assert(documents.groupBy { it["canonicalName"] }.any { it.value.size > 1 }) {
                    val wrong = documents.groupBy { it["canonicalName"] }.filter { it.value.size > 1 }
                    val tags = wrong.map { documents -> documents.value.map { it["tags"] } }
                    "Expected a unique canonical name: ${wrong.keys} [$tags]"
                }

                val structuralDocuments = documents.subList(2, documents.size)

                collection.insertMany(structuralDocuments.distinct())
            }
        }
    }

    private fun reportProjectToGradientServer(project: Project, versionId: UUID) {
        logger.debug { "Reporting the structural analysis." }

        val report = StructuralReport.newBuilder()
                .setProjectName(project.canonicalName)
                .setVersionId(versionId.bytes())
                .build()
        try {
            structuralServiceGrpc.report(report)
        } catch (e: Exception) {
            logger.error(e) { "Could not report the structural analysis: ${project.canonicalName}" }
        }
    }

    private fun versionExistsNot(project: Project): UUID? {
        var similarVersionId: UUID? = null

        serverManager.mongoDatabase(toMongoDatabaseName(project.canonicalName))?.let { db ->
            if (MongoDBConventions.PROJECT_COLLECTION in db.listCollectionNames()) {
                val collection = db.getCollection(MongoDBConventions.PROJECT_COLLECTION)
                val versionQuery = Document("tags", "Version")
                        .append("versionHash", project.version.versionHash)

                similarVersionId = collection.find(versionQuery).first()?.let {
                    if ("id" in it) it["id"] as UUID else null
                }
            } else {
                logger.error { "Expected the project collection to exist: ${project.canonicalName}" }
            }
        }

        return similarVersionId
    }
}