package at.jku.isse.gradient.dal

import at.jku.isse.gradient.GradientConfig
import at.jku.isse.gradient.GradientEvents
import at.jku.isse.gradient.MongoDBConventions
import com.google.common.eventbus.EventBus
import com.google.common.eventbus.Subscribe
import com.google.inject.Inject
import com.mongodb.ConnectionString
import com.mongodb.MongoClientSettings
import com.mongodb.client.MongoClient
import com.mongodb.client.MongoClients
import com.mongodb.client.MongoDatabase
import mu.KotlinLogging
import org.bson.UuidRepresentation
import org.bson.codecs.UuidCodec
import org.bson.codecs.configuration.CodecRegistries
import java.util.concurrent.TimeUnit

private val logger = KotlinLogging.logger {}


class ServerManager @Inject constructor(private val config: GradientConfig,
                                        gradientEventBus: EventBus) {

    private var mongoClient: MongoClient? = null

    init {
        gradientEventBus.register(this)
    }

    @Synchronized
    fun mongoDatabase(name: String): MongoDatabase? {
        require(name.isNotBlank())

        if (mongoClient == null) {
            mongoClient = initMongoClient()
        }

        if (mongoClient == null) logger.error { "Could not connect to MongoDb. Is the service running?" }
        return mongoClient?.getDatabase(toMongoDatabaseName(name))
    }


    @Synchronized
    fun backendAvailable(): Boolean {
        if (mongoClient == null) {
            mongoClient = initMongoClient()
        }

        logger.debug { "Backend available: ${mongoClient != null}" }
        return mongoClient != null
    }

    @Subscribe
    @Suppress("UNUSED_PARAMETER")
    fun shutdown(shutdown: GradientEvents.Shutdown) {
        logger.debug { "Closing the database clients." }

        mongoClient?.close()
    }

    private fun initMongoClient(): MongoClient? {
        val connectionString = System.getenv("MONGODB_CONNECTION_STRING") ?: config.mongoConnectionString()

        logger.debug { "Connecting to MongoDB: $connectionString" }

        return retry(config.retryLogicTimeout(), config.retryLogicRetries()) {
            val settings = MongoClientSettings.builder()
                    .applyToServerSettings { it.applyConnectionString(ConnectionString(config.mongoConnectionString())) }
                    .applyToSocketSettings { it.connectTimeout(100, TimeUnit.MILLISECONDS) }
                    .applyToClusterSettings { it.serverSelectionTimeout(100, TimeUnit.MILLISECONDS) }
                    .codecRegistry(CodecRegistries.fromRegistries(
                            CodecRegistries.fromCodecs(UuidCodec(UuidRepresentation.STANDARD)),
                            MongoClientSettings.getDefaultCodecRegistry()
                    ))
                    .build()
            val client = MongoClients.create(settings)
            client.listDatabaseNames().first() // trigger connection
            client
        }
    }

    private fun <T> retry(timeout: Int = 500, retries: Int = 2, action: () -> T): T? {
        return try {
            logger.debug { "Retry logic: timeout=$timeout, retries=$retries" }
            action()
        } catch (e: Exception) {
            try {
                Thread.sleep(timeout.toLong())
            } catch (e: InterruptedException) {
            }

            if (retries > 0) {
                retry(2 * timeout, retries - 1, action)
            } else {
                null
            }
        }
    }


}

fun toMongoDatabaseName(projectName: String): String {
    require(projectName.isNotBlank())

    var newName = projectName.replace(".", "-")
    if (newName.length > MongoDBConventions.MAX_DB_NAME) {
        val components = newName.split("-")
        val prefix = components.dropLast(1)
                .map { it[0] }
                .joinToString("-")
        newName = "$prefix-${components.last()}"

        assert(newName.length < MongoDBConventions.MAX_DB_NAME) { "Could not compact database name below mongo db restrictions." }
    }

    return newName
}

fun toMongoCollectionName(collectionName: String, postfix: String): String {
    require(collectionName.isNotBlank() && postfix.isNotBlank())

    val newName = collectionName.replace("$", "-")
    return "${newName.take(5)}-${newName.takeLast(5)}-$postfix"
}