package at.jku.isse.gradient.dal

import at.jku.isse.gradient.GradientConfig
import at.jku.isse.gradient.dal.mongo.MongoEventDao
import at.jku.isse.gradient.dal.mongo.MongoProjectDao
import at.jku.isse.gradient.dal.mongo.QueryMapper
import at.jku.isse.gradient.message.EventServiceGrpc
import at.jku.isse.gradient.message.StructuralServiceGrpc
import com.google.inject.AbstractModule
import com.google.inject.Provides
import com.google.inject.Scopes
import com.google.inject.Singleton
import com.google.inject.name.Names
import io.grpc.Channel
import io.grpc.ManagedChannelBuilder
import org.aeonbits.owner.ConfigFactory
import java.util.*
import java.util.concurrent.TimeUnit


class DalModule : AbstractModule() {
    override fun configure() {
        bindConfigurations()

        bindDatabases()

        bindDaos()

        bindMappers()
    }

    private fun bindMappers() {
        bind(QueryMapper::class.java)
    }

    private fun bindDatabases() {
        bind(UUID::class.java)
                .annotatedWith(Names.named("runtimeId"))
                .toInstance(UUID.randomUUID())

        bind(ServerManager::class.java).`in`(Scopes.SINGLETON)
    }

    private fun bindConfigurations() {
        val coreConfig = ConfigFactory.create(GradientConfig::class.java)
        bind(GradientConfig::class.java)
                .toInstance(coreConfig)
    }

    private fun bindDaos() {
        bind(ProjectDao::class.java).to(MongoProjectDao::class.java)
        bind(EventDao::class.java).to(MongoEventDao::class.java)
    }

    @Provides
    fun provideStructuralServiceStub(channel: Channel): StructuralServiceGrpc.StructuralServiceBlockingStub {
        return StructuralServiceGrpc.newBlockingStub(channel)
    }

    @Provides
    fun provideBehavioralServiceStub(channel: Channel): EventServiceGrpc.EventServiceBlockingStub {
        return EventServiceGrpc.newBlockingStub(channel)
    }

    @Provides
    @Singleton
    fun provideChannel(config: GradientConfig): Channel {
        val channel = ManagedChannelBuilder
                .forTarget(config.gradientServerConnectionString())
                .usePlaintext()
                .build()

        Runtime.getRuntime().addShutdownHook(Thread(Runnable {
            channel.shutdown()
            channel.awaitTermination(10, TimeUnit.SECONDS)
        }))

        return channel
    }
}