package at.jku.isse.gradient.service

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.Util
import at.jku.isse.gradient.dal.ProjectDao
import at.jku.isse.gradient.lang.java.ModelComponents
import at.jku.isse.gradient.lang.java.ModelTransformer
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.model.Version
import at.jku.isse.gradient.model.Versionable
import at.jku.isse.gradient.profiledDebug
import at.jku.isse.gradient.profiledInfo
import com.google.common.hash.HashCode
import com.google.common.hash.Hashing
import com.google.inject.Inject
import mu.KotlinLogging
import spoon.Launcher
import spoon.reflect.factory.Factory
import spoon.support.compiler.FileSystemFile
import spoon.support.compiler.FileSystemFolder
import spoon.support.compiler.ZipFolder
import java.nio.file.Files
import java.nio.file.Paths
import java.util.*

private val logger = KotlinLogging.logger {}


class StructuralService
@Inject internal constructor(private val projectDao: ProjectDao,
                             private val modelTransformer: ModelTransformer) {

    companion object {
        private val emptyHash = HashCode.fromInt(0)
    }

    fun analyze(groupName: String,
                projectName: String,
                sourcePaths: Collection<String>,
                classPaths: Collection<String> = listOf()): Project {
        require(groupName.isNotBlank())
        require(projectName.isNotBlank())
        require(sourcePaths.isNotEmpty())

        val canonicalProjectName = "$groupName.$projectName"

        logger.profiledDebug { "Building model: $canonicalProjectName" }

        val modelComponents = buildModel(sourcePaths, classPaths)

        logger.profiledInfo(ProfilerState.RESET) { "Finished building the model: ${modelComponents.types.size} types" }

        val version = createVersion(modelComponents)
        val project = Project(Util.uuid(), projectName, canonicalProjectName, version)

        logger.profiledDebug(ProfilerState.RESET) { "Persisting the model" }

        projectDao.storeProject(project)

        logger.profiledInfo(ProfilerState.STOP) { "Finished persisting the model" }

        return project
    }

    private fun buildModel(inputPaths: Collection<String>,
                           classPaths: Collection<String>): ModelComponents {
        val spoonModel = buildSpoonModel(inputPaths, classPaths)
        return modelTransformer.transform(spoonModel, ModelTransformer.TypeUniverse.DECLARED)
    }

    private fun createVersion(modelComponents: ModelComponents): Version {
        val typeHash = modelComponents.types.keys
                .sorted()
                .map { modelComponents.types[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val propertyHash = modelComponents.properties.keys
                .sorted()
                .map { modelComponents.properties[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val executableHash = modelComponents.executables.keys
                .sorted()
                .map { modelComponents.executables[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val versionHash = listOf(typeHash, propertyHash, executableHash)
                .filter { it != emptyHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else emptyHash }

        val versionables = modelComponents
                .filterIsInstance(Versionable::class.java)
                .toMutableSet()

        // Use strong uuid for versions
        return Version(UUID.randomUUID(), versionHash.asBytes(), versionables)
    }

    private fun buildSpoonModel(inputPaths: Collection<String>, classPaths: Collection<String>): Factory {
        return Launcher().let { launcher ->
            inputPaths
                    .distinct()
                    .map { Paths.get(it) }
                    .filter { Files.exists(it) }
                    .forEach {
                        val resource = when {
                            Files.isDirectory(it) -> FileSystemFolder(it.toFile())
                            it.toString().endsWith(".jar") -> ZipFolder(it.toFile())
                            else -> FileSystemFile(it.toFile())
                        }

                        launcher.addInputResource(resource)
                    }

            if (classPaths.isNotEmpty()) launcher.environment.sourceClasspath = classPaths.toTypedArray()
            launcher.environment.noClasspath = classPaths.isEmpty()
            launcher.buildModel()
            launcher.factory
        }
    }
}