package at.jku.isse.gradient.lang.java

import com.google.common.base.Stopwatch
import mu.KotlinLogging
import spoon.reflect.factory.Factory
import spoon.reflect.reference.CtTypeReference
import java.util.stream.Collectors

private val logger = KotlinLogging.logger {}

class ModelTransformer {

    fun transform(factory: Factory, universe: TypeUniverse = TypeUniverse.DECLARED): ModelComponents {
        val declarationCache = DeclarationCache()
        val nodeTransformer = AsgNodeTransformer(declarationCache)
        val relationshipTransformer = AsgRelationshipTransformer(declarationCache)

        val stopWatch = Stopwatch.createStarted()

        logger.debug { "Preparing type universe with upper limit: $universe" }
        val (declaredTypes, importedTypes, fringeType) = prepareTypes(factory, universe)
        logger.debug { "\tFinished after $stopWatch" }

        stopWatch.reset().start()
        logger.debug { "Transforming ASG elements to nodes" }
        val model = (declaredTypes.values + importedTypes.values + fringeType.values)
                .parallelStream()
                .map(nodeTransformer::transform)
                .collect(Collectors.toList())
                .fold(ModelComponents()) { total, next -> ModelComponents.mergeLeft(total, next) }
        logger.debug { "\tFinished after $stopWatch" }
        logger.debug { "\tGenerated $model" }

        stopWatch.reset().start()
        logger.debug { "Attaching relationships to nodes" }
        (declaredTypes.values + importedTypes.values)
                .parallelStream()
                .forEach { relationshipTransformer.transform(it, model) }

        logger.debug { "\tFinished after $stopWatch" }
        logger.debug { "\tGenerated $model" }
        logger.debug { "Cache Statistics:\n" + declarationCache.toString().prependIndent("\t\t") }

        return model
    }

    private fun prepareTypes(factory: Factory, universe: TypeUniverse): Types {
        val typeFactory = factory.Type()

        val excludedTypes = { t: CtTypeReference<*> ->
            !t.simpleName.isBlank() &&
                    t != factory.Type().NULL_TYPE &&
                    !t.isGenerics
        }

        val builtInTypes = listOf<CtTypeReference<*>>(typeFactory.BOOLEAN, typeFactory.BYTE, typeFactory.CHARACTER,
                typeFactory.DATE, typeFactory.DOUBLE, typeFactory.FLOAT, typeFactory.INTEGER, typeFactory.LONG,
                typeFactory.SHORT, typeFactory.STRING, typeFactory.VOID, typeFactory.OBJECT, typeFactory.ITERABLE,
                typeFactory.COLLECTION)
                .associateBy { it.canonicalName() }


        val declaredTypes = factory.Type().getAll(true)
                .map { it.reference }
                .associateBy { it.canonicalName() }

        val importedTypes = mutableMapOf<String, CtTypeReference<*>>()
        val fringeTypes = mutableMapOf<String, CtTypeReference<*>>()

        if (universe >= TypeUniverse.IMPORTED) {
            importedTypes += declaredTypes.values
                    .flatMap { it.typeDeclaration.referencedTypes }
                    .filter(excludedTypes)
                    .filterNotNull()
                    .associateBy { it.canonicalName() }
                    .toMutableMap()

            importedTypes += builtInTypes
            importedTypes -= declaredTypes.keys

            if (universe >= TypeUniverse.FRINGE) {
                fringeTypes += importedTypes.values
                        .flatMap { it.typeDeclaration.referencedTypes }
                        .filter(excludedTypes)
                        .filterNotNull()
                        .associateBy { it.canonicalName() }
                        .toMutableMap()

                fringeTypes -= declaredTypes.keys + importedTypes.keys
            }
        }

        return Types(declaredTypes = declaredTypes, importedTypes = importedTypes, fringeType = fringeTypes)
    }

    private data class Types(val declaredTypes: Map<String, CtTypeReference<*>>,
                             val importedTypes: Map<String, CtTypeReference<*>>,
                             val fringeType: Map<String, CtTypeReference<*>>)

    enum class TypeUniverse {
        DECLARED, IMPORTED, FRINGE
    }
}



