package at.jku.isse.gradient.dal.mongo

import at.jku.isse.gradient.model.*
import mu.KotlinLogging
import org.bson.Document
import java.util.*

private val logger = KotlinLogging.logger {}

class QueryMapper {

    fun mapAll(project: Project): List<Document> {
        return Transformer().run {
            scan(project)
            documents
        }
    }

    fun map(project: Project): Document {
        return Document("tags", listOf("Project"))
                .append("id", project.id)
                .append("name", project.name)
                .append("canonical_name", project.canonicalName)
    }

    fun map(version: Version): Document {
        return Document("tags", listOf("Version"))
                .append("id", version.id)
                .append("version_hash", version.versionHash)
                .append("timestamp", version.timestamp)
    }

    fun map(type: Type): Document {
        return Document("tags", listOf("Versionable", "Type"))
                .append("id", type.id)
                .append("name", type.name)
                .append("canonical_name", type.canonicalName)
                .append("availability", type.availability.name)
                .append("quality", type.quality.name)
                .append("visibility", type.visibility.name)
                .append("is_generics", type.isGenerics)
                .append("is_gradient_model", type.isGradientModel)
                .append("super_types", type.isExtending.map { it.id })
                .append("version_hash", type.versionHash.asBytes())
    }

    fun map(property: Property, declaringTypeId: UUID): Document {
        val document = Document("tags", listOf("Versionable", "Property"))
                .append("id", property.id)
                .append("name", property.name)
                .append("canonical_name", property.canonicalName)
                .append("visibility", property.visibility.name)
                .append("is_static", property.isStatic)
                .append("is_immutable", property.isImmutable)
                .append("gradient_type", property.gradientType.name)
                .append("version_hash", property.versionHash.asBytes())
                .append("shadows", property.shadows.map { it.id })
                .append("declaring_type", declaringTypeId)

        property.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(executable: Executable, declaringTypeId: UUID): Document {
        val document = Document("tags", listOf("Versionable", "Executable"))
                .append("id", executable.id)
                .append("name", executable.name)
                .append("canonical_name", executable.canonicalName)
                .append("is_constructor", executable.isConstructor)
                .append("is_abstract", executable.isAbstract)
                .append("is_static", executable.isStatic)
                .append("visibility", executable.visibility.name)
                .append("accessor_type", executable.accessor.name)
                .append("gradient_type", executable.gradientType.name)
                .append("version_hash", executable.versionHash.asBytes())
                .append("invokes", executable.invokes.map { it.id })
                .append("accesses", executable.accesses.map { it.id })
                .append("overrides", executable.overrides.map { it.id })
                .append("declaring_type", declaringTypeId)

        executable.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(parameter: Parameter, declaringExecutableId: UUID): Document {
        val document = Document("tags", listOf("Parameter"))
                .append("id", parameter.id)
                .append("name", parameter.name)
                .append("canonical_name", parameter.canonicalName)
                .append("declaring_executable", declaringExecutableId)
                .append("gradient_type", parameter.gradientType.name)

        parameter.type?.let { document.append("type", it.id) }

        return document
    }

    fun map(typeParameterMapping: TypeParameterMapping, declaringElement: UUID): Document {
        return Document("tags", listOf("TypeParameter"))
                .append("id", typeParameterMapping.id)
                .append("canonical_name", typeParameterMapping.canonicalName)
                .append("actual_type", typeParameterMapping.actualType.id)
                .append("parameter", typeParameterMapping.parameter.id)
                .append("declaring_element", declaringElement)
    }

    fun map(elementType: ElementType): Document {
        return Document("tags", listOf("ElementType"))
                .append("id", elementType.id)
                .append("canonical_name", elementType.canonicalName)
                .append("cardinality", elementType.cardinality.name)
                .append("type_of", elementType.typeOf.id)
                .append("mappings", elementType.typeParameterMappings.map { it.id })
    }

    fun map(access: Access): Document {
        val document = Document("tags", listOf("Access"))
                .append("id", access.id)
                .append("canonical_name", access.canonicalName)
                .append("source", access.source.id)
                .append("target", access.target.id)
                .append("cardinality", access.cardinality.name)
                .append("access_type", access.accessType.name)

        access.delegate?.let { document.append("via", access.delegate.id) }

        return document
    }

    fun map(invocation: Invocation): Document {
        val document = Document("tags", listOf("Invocation"))
                .append("id", invocation.id)
                .append("canonical_name", invocation.canonicalName)
                .append("source", invocation.source.id)
                .append("target", invocation.target.id)
                .append("cardinality", invocation.cardinality.name)

        invocation.delegate?.let { document.append("via", it.id) }

        return document
    }

    fun map(overriding: Overriding): Document {
        return Document("tags", listOf("Overriding"))
                .append("id", overriding.id)
                .append("canonical_name", overriding.canonicalName)
                .append("source", overriding.source.id)
                .append("target", overriding.target.id)
                .append("quality", overriding.overridingQuality.name)
    }

    private inner class Transformer : ModelVisitor {

        val visitedElements = mutableSetOf<UUID>()
        val documents: MutableList<Document> = mutableListOf()

        var parentIds: Stack<UUID> = Stack()

        override fun scan(entity: ModelVisitable) {
            assert(entity is Project || parentIds.isNotEmpty()) { "Parent ids is empty, did you call pushPopScan?" }
            if (entity is Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        private fun pushPopScan(e: Entity) {
            parentIds.push(e.id)
            scan(e)
            val id = parentIds.pop()

            assert(id == e.id) { "Something when wrong on the id stack. Did you forget to push or pop?" }
        }

        private fun pushPopScan(e: Iterable<Entity>) {
            e.forEach { pushPopScan(it) }
        }

        override fun visitProject(e: Project) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents.add(map(e))

            pushPopScan(e.version)
        }

        override fun visitVersion(e: Version) {
            documents.add(map(e))

            pushPopScan(e.revisions)
        }

        override fun visitType(e: Type) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            documents.add(map(e))

            pushPopScan(e.properties)
            pushPopScan(e.inheritedProperties)
            pushPopScan(e.executables)
            pushPopScan(e.inheritedExecutables)
            pushPopScan(e.typeParameters)
            pushPopScan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            documents.add(map(e, parentIds.peek()))

            pushPopScan(e.shadows)
            e.type?.let { pushPopScan(it) }
        }

        override fun visitExecutable(e: Executable) {
            documents.add(map(e, parentIds.peek()))

            pushPopScan(e.parameters)
            pushPopScan(e.returns)
            pushPopScan(e.overrides)
            e.type?.let { pushPopScan(it) }
            pushPopScan(e.invokes)
            pushPopScan(e.accesses)
            pushPopScan(e.typeParameters)
        }

        override fun visitParameter(e: Parameter) {
            documents.add(map(e, parentIds.peek()))

            e.type?.let { pushPopScan(it) }
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            documents.add(map(e, parentIds.peek()))

            pushPopScan(e.actualType)
            pushPopScan(e.parameter)
        }

        override fun visitElementType(e: ElementType) {
            documents.add(map(e))

            pushPopScan(e.typeOf)
            pushPopScan(e.typeParameterMappings)
        }

        override fun visitAccess(e: Access) {
            documents.add(map(e))

            pushPopScan(e.source)
            pushPopScan(e.target)
            e.delegate?.let { pushPopScan(it) }
        }

        override fun visitInvocation(e: Invocation) {
            documents.add(map(e))

            pushPopScan(e.source)
            pushPopScan(e.target)
            e.delegate?.let { pushPopScan(it) }
        }

        override fun visitOverriding(e: Overriding) {
            documents.add(map(e))

            pushPopScan(e.source)
            pushPopScan(e.target)
        }
    }


}