package at.jku.isse.gradient.dal.mongo

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.Util
import at.jku.isse.gradient.dal.ProjectDao
import at.jku.isse.gradient.dal.ServerManager
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.profiledDebug
import at.jku.isse.gradient.profiledTrace
import at.jku.isse.psma.gradient.server.StructuralReport
import at.jku.isse.psma.gradient.server.StructuralServiceGrpc
import com.google.inject.Inject
import com.mongodb.client.MongoCollection
import com.mongodb.client.MongoDatabase
import com.mongodb.client.model.Filters
import com.mongodb.client.model.IndexOptions
import com.mongodb.client.model.Indexes
import mu.KotlinLogging
import org.bson.Document
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}

fun <T> Document.getList(key: String): List<T> {
    @Suppress("UNCHECKED_CAST")
    return get(key) as List<T>
}

class MongoProjectDao
@Inject constructor(private val serverManager: ServerManager,
                    private val queryMapper: QueryMapper,
                    private val structuralServiceGrpc: StructuralServiceGrpc.StructuralServiceBlockingStub) : ProjectDao {

    override fun storeProject(project: Project) {
        logger.profiledTrace { "Storing project: ${project.canonicalName}" }

        if (project.version.revisions.isEmpty()) {
            logger.debug { "Ignoring revision as it does not contain any elements: ${project.canonicalName}@${project.version.id}" }
        } else {
            val similarVersionId = versionExistsNot(project)
            if (similarVersionId != null) {
                logger.debug { "Found version with similar versionHash ($similarVersionId). Ignoring currently processed version." }
            } else {
                storeProjectCollection(project)
                storeStructuralCollection(project)
                reportProjectToGradientServer(project)
            }
        }

        logger.profiledDebug(ProfilerState.STOP) { "Finished storing project:${project.canonicalName}" }
    }

    private fun versionExistsNot(project: Project): UUID? {
        var similarVersionId: UUID? = null

        serverManager.mongoDatabase(toMongoDatabaseName(project.canonicalName))?.let { db ->
            val collection = getOrCreateProjectCollection(db)

            val versionQuery = Document("tags", "Version")
                    .append("versionHash", project.version.versionHash)

            similarVersionId = collection.find(versionQuery).first()?.let {
                if ("id" in it) it["id"] as UUID else null
            }
        }

        return similarVersionId
    }

    private fun storeProjectCollection(project: Project) {
        serverManager.mongoDatabase(project.canonicalName)?.let { db ->
            val collection = getOrCreateProjectCollection(db)

            val projectQuery = Document("tags", "Project")
                    .append("canonicalName", project.canonicalName)

            val result = collection.find(projectQuery).first()
            if (result == null) {
                collection.insertOne(queryMapper.map(project))
            }

            collection.insertOne(queryMapper.map(project.version))
        }
    }

    private fun storeStructuralCollection(project: Project) {
        serverManager.mongoDatabase(project.canonicalName)?.let { db ->
            val collectionName = toCollectionName(project.version.id.toString(), STRUCTURAL_COLLECTION_POSTFIX)
            val collection = getOrCreateStructuralCollection(db, collectionName)

            assert(collection.find().first() == null) { "Expecting to hav created a new version." }
            val documents = queryMapper.mapAll(project)

            assert("Project" in documents[0].getList<String>("tags")) { "Expected the project to be the first document." }
            assert("Version" in documents[1].getList<String>("tags")) { "Expected the version to be the first document." }
            assert(documents.groupBy { it["canonicalName"] }.any { it.value.size > 1 }) {
                val wrong = documents.groupBy { it["canonicalName"] }.filter { it.value.size > 1 }
                val tags = wrong.map { documents -> documents.value.map { it["tags"] } }
                "Expected a unique canonical name: ${wrong.keys} [$tags]"
            }

            val structuralDocuments = documents.subList(2, documents.size)

            collection.insertMany(structuralDocuments.distinct())

            reportProjectToGradientServer(project)
        }
    }

    private fun reportProjectToGradientServer(project: Project) {
        logger.debug { "Reporting the structural analysis." }

        val report = StructuralReport.newBuilder()
                .setId(Util.uuid().toString())
                .setProjectId(project.id.toString())
                .setProjectName(project.name)
                .setProjectCanonicalName(project.canonicalName)
                .setTimestamp(Instant.now().epochSecond)
                .build()
        try {
            structuralServiceGrpc.report(report)
        } catch (e: Exception) {
            logger.error(e) { "Could not report the structural analysis: ${project.canonicalName}" }
        }
    }

    private fun getOrCreateProjectCollection(db: MongoDatabase): MongoCollection<Document> {
        val collection: MongoCollection<Document>
        if (PROJECT_COLLECTION in db.listCollectionNames()) {
            collection = db.getCollection(PROJECT_COLLECTION)
        } else {
            logger.debug { "Creating collection: $PROJECT_COLLECTION" }

            db.createCollection(PROJECT_COLLECTION)

            collection = db.getCollection(PROJECT_COLLECTION)
            collection.createIndex(Indexes.ascending("id"), IndexOptions().unique(true))
            collection.createIndex(Indexes.ascending("tags"))
            collection.createIndex(Indexes.ascending("canonical_name"))
        }

        return collection
    }

    private fun getOrCreateStructuralCollection(db: MongoDatabase, collectionName: String): MongoCollection<Document> {
        assert(collectionName.isNotBlank())

        val collection: MongoCollection<Document>
        if (collectionName in db.listCollectionNames()) {
            collection = db.getCollection(collectionName)
        } else {
            logger.debug { "Creating collection: $collectionName" }

            db.createCollection(collectionName)

            collection = db.getCollection(collectionName)
            collection.createIndex(Indexes.ascending("id"), IndexOptions().unique(true))
            collection.createIndex(Indexes.ascending("tags"))
            collection.createIndex(Indexes.ascending("canonical_name"))
            collection.createIndex(Indexes.ascending("is_gradient_model"))
            collection.createIndex(Indexes.ascending("declaring_type"))
            collection.createIndex(Indexes.ascending("declaring_executable"))
        }

        return collection
    }
}