package at.jku.isse.gradient.dal.mongo

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.dal.ObservationDao
import at.jku.isse.gradient.dal.ServerManager
import at.jku.isse.gradient.profiledDebug
import at.jku.isse.gradient.profiledTrace
import com.google.inject.Inject
import com.mongodb.client.MongoCollection
import com.mongodb.client.MongoDatabase
import com.mongodb.client.model.Indexes
import mu.KotlinLogging
import org.bson.Document
import java.util.*

private val logger = KotlinLogging.logger {}

class MongoObservationDao
@Inject constructor(private val serverManager: ServerManager) : ObservationDao {

    override fun reportObservations(projectName: String, versionId: UUID, runtimeId: UUID,
                                    observationBatch: List<Map<String, Any>>) {
        require(projectName.isNotBlank())
        require(observationBatch.isNotEmpty())

        logger.profiledTrace { "Storing ${observationBatch.size} observations: $projectName" }

        serverManager.mongoDatabase(projectName)?.let { db ->

            val collectionName = toCollectionName(versionId.toString(), EVENT_COLLECTION_POSTFIX)
            val collection = if (collectionName in db.listCollectionNames()) {
                db.getCollection(collectionName)
            } else {
                getOrCreateBehavioralCollection(db, collectionName)
            }

            val documents = observationBatch.map { Document(it) }
            collection.insertMany(documents)
        }

        logger.profiledDebug(ProfilerState.STOP) { "Finished storing ${observationBatch.size} observations: $projectName" }
    }


    private fun getOrCreateBehavioralCollection(db: MongoDatabase, collectionName: String): MongoCollection<Document> {
        assert(collectionName.isNotBlank())

        logger.debug { "Creating behavioral collection: $collectionName" }

        db.createCollection(collectionName)

        val collection = db.getCollection(collectionName)
        collection.createIndex(Indexes.ascending("value_type"))
        collection.createIndex(Indexes.ascending("context"))
        collection.createIndex(Indexes.ascending("parent_context"))
        collection.createIndex(Indexes.compoundIndex(
                Indexes.ascending("element_name"),
                Indexes.ascending("event_type"))
        )
        return collection
    }
}