package at.jku.isse.gradient.dal.neo4j

import at.jku.isse.gradient.dal.LabelVisitor
import at.jku.isse.gradient.model.*
import com.google.common.collect.ArrayListMultimap
import com.google.common.collect.Multimap
import com.google.inject.Inject
import mu.KotlinLogging

private val logger = KotlinLogging.logger {}

class QueryRelationshipMapper @Inject constructor(private val labelVisitor: LabelVisitor) {

    fun map(entities: Iterable<at.jku.isse.gradient.model.Entity>): Multimap<String, Map<String, Any>> {
        return RelationshipVisitor()
                .run {
                    scan(entities)
                    queries
                }
    }

    private inner class RelationshipVisitor : at.jku.isse.gradient.model.ModelVisitor {

        val queries: Multimap<String, Map<String, Any>> = ArrayListMultimap.create()
        val visitedElements = mutableSetOf<String>()

        override fun scan(entity: at.jku.isse.gradient.model.ModelVisitable) {
            if (entity is at.jku.isse.gradient.model.Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        override fun visitProject(e: Project) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            e.versions.forEach { addRelationship(e, it, "contains") }

            if (e.versions.isNotEmpty()) {
                addRelationship(e, e.versions.last(), "lastRevision")
            }

            e.lastVersion?.let { visitVersion(it) }
            e.versions.forEach { visitVersion(it) }
        }

        override fun visitVersion(e: Version) {
            e.nextVersion?.let { addRelationship(e, it, "next") }
            e.revisions.forEach {
                if (it is at.jku.isse.gradient.model.CanonicalEntity) addRelationship(e, it, "contains")
                when (it) {
                    is Type -> visitType(it)
                    is Executable -> visitExecutable(it)
                    is Property -> visitProperty(it)
                }
            }
        }

        override fun visitType(e: Type) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            e.isExtending.forEach { addRelationship(e, it, "extends") }

            e.typeParameters.forEach { addRelationship(e, it, "parametrizedBy") }

            e.properties.forEach { addRelationship(e, it, "declares") }

            e.executables.forEach { addRelationship(e, it, "declares") }

            e.inheritedProperties.forEach { addRelationship(e, it, "inherits") }

            e.inheritedExecutables.forEach { addRelationship(e, it, "inherits") }

            scan(e.typeParameters)
            scan(e.isExtending)
            scan(e.properties)
            scan(e.executables)
            scan(e.inheritedProperties)
            scan(e.inheritedExecutables)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            addRelationship(e, e.parameter, "parameter")
            addRelationship(e, e.actualType, "replacedBy")

            scan(e.parameter)
            scan(e.actualType)
        }

        override fun visitProperty(e: Property) {
            e.shadows.forEach { addRelationship(e, it, "shadows") }
            scan(e.shadows)

            e.type?.let {
                addRelationship(e, it, "isOf")
                e.type?.let { scan(it) }
            }
        }

        override fun visitExecutable(e: Executable) {
            e.overrides.forEach { addRelationship(it.source, it.target, "overrides", mutableMapOf("quality" to it.overridingQuality.name)) }
            e.parameters.forEach { addRelationship(e, it, "receives") }
            e.returns.forEach { addRelationship(e, it, "returns") }
            e.typeParameters.forEach { addRelationship(e, it, "parametrizedBy") }

            e.type?.let { addRelationship(e, it, "isOf") }

            e.type?.let { scan(it) }
            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            scan(e.accesses)
            scan(e.invokes)
        }

        override fun visitParameter(e: Parameter) {
            e.type?.let {
                addRelationship(e, it, "isOf")
                scan(it)
            }
        }

        override fun visitElementType(e: ElementType) {
            e.typeParameterMappings.forEach { addRelationship(e, it, "maps") }

            addRelationship(e, e.typeOf, "typeOf")

            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitAccess(e: Access) {
            addRelationship(e, e.source, "source")
            e.delegate?.let { addRelationship(e, e.delegate, "via") }
            addRelationship(e, e.target, "target")
            scan(e.source)
            scan(e.target)
        }

        override fun visitInvocation(e: Invocation) {
            addRelationship(e, e.source, "source")
            e.delegate?.let { addRelationship(e, e.delegate, "via") }
            addRelationship(e, e.target, "target")
            scan(e.source)
            scan(e.target)
        }

        fun createQuery(sourceLabel: String,
                        label: String, parameterKeys: Set<String>,
                        targetLabel: String): String {
            val properties = parameterKeys.joinToString(", ") { "$it:row.$it" }

            return when {
                properties.isEmpty() ->
                    """
                    UNWIND {parameters} as row
                    MATCH (source$sourceLabel {id:row.sourceId}), (target$targetLabel {id:row.targetId})
                    MERGE (source) -[:$label]-> (target)
                    """.trimIndent()
                else ->
                    """
                    UNWIND {parameters} as row
                    MATCH (source$sourceLabel {id:row.sourceId}), (target$targetLabel {id:row.targetId})
                    MERGE (source) -[:$label {$properties}]-> (target)
                    """.trimIndent()
            }
        }

        fun addRelationship(source: at.jku.isse.gradient.model.Entity,
                            target: at.jku.isse.gradient.model.Entity,
                            label: String,
                            properties: MutableMap<String, Any> = mutableMapOf()) {
            val id = "(${source.id}) -[:$label]-> (${target.id})"

            if (id !in visitedElements) {
                val query = createQuery(labelVisitor.labelOf(source), label, properties.keys, labelVisitor.labelOf(target))
                properties["sourceId"] = source.id
                properties["targetId"] = target.id

                queries.put(query, properties)
                visitedElements.add(id)
            }
        }
    }
}