package at.jku.isse.gradient.dal.neo4j

import at.jku.isse.gradient.dal.LabelVisitor
import at.jku.isse.gradient.model.*
import com.google.common.collect.ArrayListMultimap
import com.google.common.collect.Multimap
import com.google.inject.Inject
import mu.KotlinLogging

private val logger = KotlinLogging.logger {}

class QueryNodeMapper @Inject constructor(private val labelVisitor: LabelVisitor) {

    fun map(entities: Iterable<at.jku.isse.gradient.model.Entity>): Multimap<String, Map<String, Any>> {
        return QueryVisitor()
                .run {
                    scan(entities)
                    queries
                }
    }

    private inner class QueryVisitor : at.jku.isse.gradient.model.ModelVisitor {

        val queries: Multimap<String, Map<String, Any>> = ArrayListMultimap.create()
        val visitedElements = mutableSetOf<String>()

        override fun scan(entity: at.jku.isse.gradient.model.ModelVisitable) {
            if (entity is at.jku.isse.gradient.model.Entity) {
                if (entity.id !in visitedElements) {
                    visitedElements.add(entity.id)
                    super.scan(entity)
                }
            }
        }

        fun createQuery(label: String, parameters: Map<String, Any>): String {
            val properties = parameters.keys
                    .joinToString(", ") { "$it:row.$it" }

            return when {
                properties.isEmpty() ->
                    """
                    UNWIND {parameters} as row
                    MERGE ($label)'
                    """.trimIndent()
                else ->
                    """
                    UNWIND {parameters} as row
                    MERGE ($label {$properties})
                    """.trimIndent()
            }
        }

        override fun visitProject(e: Project) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.versions)
        }

        override fun visitVersion(e: Version) {
            val parameters = mutableMapOf("id" to e.id)

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.revisions)
            e.nextVersion?.let { scan(it) }
        }

        override fun visitType(e: Type) {
            logger.debug { "Mapping: ${e.canonicalName}" }

            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name,
                    "availability" to e.availability.name,
                    "quality" to e.quality.name,
                    "visibility" to e.visibility.name,
                    "isGenerics" to e.isGenerics,
                    "isGradientModel" to e.isGradientModel,
                    "versionHash" to e.versionHash.asLong()
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.properties)
            scan(e.inheritedProperties)
            scan(e.executables)
            scan(e.inheritedExecutables)
            scan(e.typeParameters)
            scan(e.isExtending)
        }

        override fun visitProperty(e: Property) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name,
                    "visibility" to e.visibility.name,
                    "isStatic" to e.isStatic,
                    "isImmutable" to e.isImmutable,
                    "gradientType" to e.gradientType.name,
                    "versionHash" to e.versionHash.asLong()
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.shadows)
            e.type?.let { scan(it) }
        }

        override fun visitExecutable(e: Executable) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name,
                    "isAbstract" to e.isAbstract,
                    "visibility" to e.visibility.name,
                    "isStatic" to e.isStatic,
                    "isConstructor" to e.isConstructor,
                    "accessorType" to e.accessor.name,
                    "gradientType" to e.gradientType.name,
                    "versionHash" to e.versionHash.asLong()
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.parameters)
            scan(e.returns)
            scan(e.overrides)
            e.type?.let { scan(it) }
            scan(e.invokes)
            scan(e.accesses)
            scan(e.typeParameters)
        }

        override fun visitInvocation(e: Invocation) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "cardinality" to e.cardinality.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.source)
            scan(e.target)
            e.delegate?.let { scan(it) }
        }

        override fun visitAccess(e: Access) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "cardinality" to e.cardinality.name,
                    "accessType" to e.accessType.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.source)
            scan(e.target)
        }

        override fun visitParameter(e: Parameter) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name,
                    "gradientType" to e.gradientType.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            e.type?.let { scan(it) }
        }

        override fun visitElementType(e: ElementType) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "cardinality" to e.cardinality.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.typeOf)
            scan(e.typeParameterMappings)
        }

        override fun visitTypeParameterMapping(e: TypeParameterMapping) {
            val parameters = mutableMapOf(
                    "id" to e.id,
                    "canonicalName" to e.canonicalName,
                    "name" to e.name
            )

            queries.put(createQuery(labelVisitor.labelOf(e), parameters), parameters)

            scan(e.actualType)
            scan(e.parameter)
        }
    }
}