package at.jku.isse.gradient.dal

import at.jku.isse.gradient.GradientConfig
import com.google.common.eventbus.EventBus
import com.google.common.eventbus.Subscribe
import com.google.inject.Inject
import com.google.inject.name.Named
import com.mongodb.ConnectionString
import com.mongodb.MongoClientSettings
import com.mongodb.client.MongoClient
import com.mongodb.client.MongoClients
import com.mongodb.client.MongoDatabase
import mu.KotlinLogging
import org.neo4j.driver.internal.logging.ConsoleLogging
import org.neo4j.driver.v1.*
import java.util.concurrent.TimeUnit
import java.util.logging.Level

private val logger = KotlinLogging.logger {}

class GradientServerManager @Inject constructor(private val config: GradientConfig,
                                                @Named("cleanupBus") cleanupBus: EventBus) {

    init {
        cleanupBus.register(this)
    }

    companion object {
        private var mongoClient: MongoClient? = null
        private var neo4jClient: Driver? = null
    }

    @Synchronized
    fun mongoDatabase(name: String, timeout: Int = 1000, retries: Int = 0): MongoDatabase? {
        if (mongoClient == null) {
            logger.debug { "Connecting to MongoDb" }
            mongoClient = initMongoClient(timeout, retries)
        }

        if (mongoClient == null) logger.error { "Could not connect to MongoDb. Is the service running?" }
        return mongoClient?.let {
            if (it.listDatabaseNames().contains(name)) {
                it.getDatabase(name)
            } else {
                logger.error { "The provided database could not be found." }
                null
            }
        }
    }

    @Synchronized
    fun neo4jSession(timeout: Int = 1000, retries: Int = 0): Session? {
        if (neo4jClient == null) {
            neo4jClient = initNeo4jClient(timeout, retries)
        }

        if (neo4jClient == null) logger.error { "Could not connect to Neo4j. Is the service running?" }
        return neo4jClient?.session()
    }

    @Synchronized
    fun backendAvailable(): Boolean {
        if (neo4jClient == null) {
            neo4jClient = initNeo4jClient(100, 0)
        }
        if (mongoClient == null) {
            mongoClient = initMongoClient(100, 0)
        }

        return neo4jClient != null && mongoClient != null
    }

    private fun initNeo4jClient(timeout: Int, retries: Int): Driver? {
        logger.debug { "Connecting to Neo4j" }
        return retry(timeout, retries, this::createNeo4jDriver)
    }

    private fun initMongoClient(timeout: Int, retries: Int): MongoClient? {
        logger.debug { "Connecting to MongoDB" }
        return retry(timeout, retries) {
            // TODO inject properties
            val settings = MongoClientSettings.builder()
                    .applyToServerSettings { it.applyConnectionString(ConnectionString(config.mongoConnectionString())) }
                    .applyToSocketSettings { it.connectTimeout(100, TimeUnit.MILLISECONDS) }
                    .build()
            val client = MongoClients.create(settings)
            client.listDatabaseNames() // trigger connection
            client
        }
    }

    @Subscribe
    @Suppress("UNUSED_PARAMETER")
    internal fun cleanup(cleanup: Cleanup) {
        mongoClient?.close()
        neo4jClient?.close()
    }

    private fun <T> retry(timeout: Int = 500, retries: Int = 2, action: () -> T): T? {
        return try {
            logger.debug { "Retry logic: timeout=$timeout, retries=$retries" }
            action()
        } catch (e: Exception) {
            try {
                Thread.sleep(timeout.toLong())
            } catch (e: InterruptedException) {
            }

            if (retries > 0) {
                retry(2 * timeout, retries - 1, action)
            } else {
                null
            }
        }
    }

    private fun createNeo4jDriver(): Driver {
        val driverConfig = Config.build()
                .withMaxTransactionRetryTime(30, TimeUnit.SECONDS)
                .withConnectionTimeout(1, TimeUnit.SECONDS)
                .withLogging(ConsoleLogging(Level.parse(config.neo4jLogLevel())))
                .toConfig()

        val connectionString = System.getenv("NEO4J_CONNECTION_STRING") ?: config.neo4jConnectionString()

        return GraphDatabase.driver(
                connectionString,
                AuthTokens.basic(config.neo4jUsername(), config.neo4jPassword()),
                driverConfig
        )
    }
}