package at.jku.isse.gradient

import com.google.common.base.Stopwatch
import mu.KLogger

enum class ProfilerState {
    START, RESET, STOP
}

private val stopwatches = object : ThreadLocal<MutableMap<KLogger, Stopwatch>>() {
    override fun initialValue(): MutableMap<KLogger, Stopwatch> {
        return mutableMapOf()
    }
}

fun KLogger.profiledInfo(state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    log(this, this::info, state, msg)
}

fun KLogger.profiledDebug(state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    log(this, this::debug, state, msg)
}

private fun log(logger: KLogger, logMethod: (() -> Any?) -> Unit, state: ProfilerState = ProfilerState.START, msg: () -> Any?) {
    when (state) {
        ProfilerState.START -> {
            stopwatches.get().put(logger, Stopwatch.createStarted())
            logMethod(msg)
        }
        ProfilerState.RESET -> {
            val stopwatch = stopwatches.get().getOrPut(logger) {
                Stopwatch.createStarted()
            }

            logMethod { "\tFinished after: $stopwatch" }
            logMethod(msg)
            stopwatch.reset().start()
        }
        ProfilerState.STOP -> {
            stopwatches.get().remove(logger)?.let {
                logMethod { "\tFinished after: ${it.stop()}" }
            }
            logMethod(msg)
        }

    }
}