package at.jku.isse.gradient.service

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.java.ModelComponents
import at.jku.isse.gradient.java.ModelTransformer
import at.jku.isse.gradient.dal.StructuralDao
import at.jku.isse.gradient.model.Version
import at.jku.isse.gradient.model.Versionable
import at.jku.isse.gradient.profiledInfo
import at.jku.isse.psma.gradient.server.StructuralReport
import at.jku.isse.psma.gradient.server.StructuralServiceGrpc
import com.fasterxml.uuid.Generators
import com.google.common.hash.HashCode
import com.google.common.hash.Hashing
import com.google.inject.Inject
import com.google.inject.name.Named
import com.google.protobuf.Empty
import io.grpc.stub.StreamObserver
import mu.KotlinLogging
import spoon.Launcher
import spoon.reflect.factory.Factory
import spoon.support.compiler.FileSystemFile
import spoon.support.compiler.FileSystemFolder
import spoon.support.compiler.ZipFolder
import java.io.File
import java.io.FileWriter
import java.nio.file.Files
import java.nio.file.Paths
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}


class StructuralService
@Inject internal constructor(private val structuralDao: StructuralDao,
                             private val structuralServiceGrpc: StructuralServiceGrpc.StructuralServiceStub,
                             private val modelTransformer: ModelTransformer,
                             @Named("structural.properties") private val structuralCacheFile: File) {

    companion object {
        private val emptyHash = HashCode.fromInt(0)
    }

    private val uuidGenerator = Generators.timeBasedGenerator()

    fun analyze(groupName: String,
                projectName: String,
                inputPaths: Iterable<String>) {

        logger.profiledInfo { "[$groupName - $projectName] Getting or registering the project." }
        val canonicalProjectName = "$groupName.$projectName"
        val projectId = structuralDao.getOrPutProject(projectName, canonicalProjectName)

        logger.profiledInfo(ProfilerState.RESET) { "[$groupName - $projectName] Parsing sources and building spoon model" }
        val spoonModel = buildSpoonModel(inputPaths)

        logger.profiledInfo(ProfilerState.RESET) { "[$groupName - $projectName] Building pillar model" }
        val modelComponents = modelTransformer.transform(spoonModel, ModelTransformer.TypeUniverse.IMPORTED)


        logger.profiledInfo(ProfilerState.RESET) { "[$groupName - $projectName] Persisting model" }
        val newVersion = createVersion(modelComponents)
        structuralDao.appendNewVersion(projectId, newVersion)

        reportAnalysis(projectId, canonicalProjectName)
        structuralProperties(canonicalProjectName, newVersion.id)

        logger.profiledInfo(ProfilerState.STOP) { "[$groupName - $projectName] Finished structural analysis" }
    }

    private fun reportAnalysis(projectId: String, projectName: String) {
        val report = StructuralReport.newBuilder()
                .setId(uuidGenerator.generate().toString())
                .setProjectId(projectId)
                .setProjectName(projectName)
                .setTimestamp(Instant.now().epochSecond)
                .build()

        structuralServiceGrpc.report(report, object : StreamObserver<Empty> {
            override fun onNext(value: Empty) {
            }

            override fun onError(t: Throwable) {
                logger.error { "The gradient-server could not receives the structural analysis report." }
            }

            override fun onCompleted() {
                logger.info { "Reported the structural analysis report to the gradient-server." }
            }
        })
    }

    private fun structuralProperties(projectName: String, versionId: String) {
        val prop = Properties()
        prop.setProperty("project.name", projectName)
        prop.setProperty("project.version", versionId)
        prop.store(FileWriter(structuralCacheFile), null)
    }

    private fun createVersion(modelComponents: ModelComponents): Version {

        val typeHash = modelComponents.types.keys
                .sorted()
                .map { modelComponents.types[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else HashCode.fromInt(0) }

        val propertyHash = modelComponents.properties.keys
                .sorted()
                .map { modelComponents.properties[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else HashCode.fromInt(0) }

        val executableHash = modelComponents.executables.keys
                .sorted()
                .map { modelComponents.executables[it]!!.versionHash }
                .let { if (it.isNotEmpty()) Hashing.combineOrdered(it) else HashCode.fromInt(0) }

        val versionHash = listOf(typeHash, propertyHash, executableHash)
                .filter { it != emptyHash }
                .let { Hashing.combineOrdered(it).toString() }

        return Version(
                versionHash,
                modelComponents
                        .filterIsInstance(Versionable::class.java)
                        .toMutableSet()
        )
    }

    private fun buildSpoonModel(inputPaths: Iterable<String>): Factory {
        return Launcher().let { launcher ->
            inputPaths
                    .distinct()
                    .map { Paths.get(it) }
                    .filter { Files.exists(it) }
                    .forEach {
                        val resource = when {
                            Files.isDirectory(it) -> FileSystemFolder(it.toFile())
                            it.toString().endsWith(".jar") -> ZipFolder(it.toFile())
                            else -> FileSystemFile(it.toFile())
                        }

                        launcher.addInputResource(resource)
                    }
            launcher.environment.noClasspath = true
            launcher.buildModel()
            launcher.factory
        }
    }
}