package at.jku.isse.gradient.java

import com.google.common.cache.CacheBuilder
import com.google.common.cache.CacheLoader
import com.google.common.reflect.ClassPath
import spoon.reflect.declaration.CtExecutable
import spoon.reflect.declaration.CtField
import spoon.reflect.declaration.CtParameter
import spoon.reflect.declaration.CtType
import spoon.reflect.factory.Factory
import spoon.reflect.reference.CtExecutableReference
import spoon.reflect.reference.CtFieldReference
import spoon.reflect.reference.CtParameterReference
import spoon.reflect.reference.CtTypeReference
import java.util.*
import java.util.concurrent.TimeUnit

class DeclarationCache(factory: Factory) {
    private val typeFactory = factory.Type()
    private val classPath: Map<String, ClassPath.ClassInfo> = ClassPath.from(factory.environment.inputClassLoader)
            .allClasses
            .associateBy { it.name }

    internal val type = CacheBuilder.newBuilder()
            .maximumSize(5000)
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .recordStats()
            .build(object : CacheLoader<CtTypeReference<*>, Optional<CtType<*>>>() {
                override fun load(key: CtTypeReference<*>): Optional<CtType<*>> {
                    if (typeFactory.get<Any>(key.qualifiedName) == null && key.qualifiedName !in classPath) return Optional.empty()
                    return Optional.of(key.typeDeclaration)
                }
            })

    internal val property = CacheBuilder.newBuilder()
            .maximumSize(5000)
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .recordStats()
            .build(object : CacheLoader<CtFieldReference<*>, Optional<CtField<*>>>() {
                override fun load(key: CtFieldReference<*>): Optional<CtField<*>> {
                    val type = key.declaringType

                    return if (type == null ||
                            typeFactory.get<Any>(type.qualifiedName) == null &&
                                    type.qualifiedName !in classPath) {
                        Optional.empty()
                    } else {
                        Optional.ofNullable(key.fieldDeclaration)
                    }
                }
            })
    internal val executable = CacheBuilder.newBuilder()
            .maximumSize(10000)
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .recordStats()
            .build(object : CacheLoader<CtExecutableReference<*>, Optional<CtExecutable<*>>>() {
                override fun load(key: CtExecutableReference<*>): Optional<CtExecutable<*>> {
                    val type = key.declaringType

                    return if (type == null ||
                            typeFactory.get<Any>(type.qualifiedName) == null &&
                                    type.qualifiedName !in classPath) {
                        Optional.empty()
                    } else {
                        Optional.ofNullable(key.executableDeclaration)
                    }
                }
            })
    internal val parameter = CacheBuilder.newBuilder()
            .maximumSize(5000)
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .recordStats()
            .build(object : CacheLoader<CtParameterReference<*>, Optional<CtParameter<*>>>() {
                override fun load(key: CtParameterReference<*>): Optional<CtParameter<*>> {
                    return Optional.of(key.declaration)
                }
            })

    override fun toString(): String {
        return "Type\t\t${type.stats()}\n" +
                "Property\t${property.stats()}\n" +
                "Executable\t${executable.stats()}\n" +
                "Parameter\t${parameter.stats()}"
    }
}

