package at.jku.isse.gradient.java

import at.jku.isse.gradient.model.*
import com.google.common.base.Stopwatch
import mu.KotlinLogging
import spoon.reflect.code.*
import spoon.reflect.declaration.*
import spoon.reflect.reference.*
import spoon.reflect.visitor.filter.TypeFilter
import spoon.support.reflect.code.CtArrayReadImpl

private val logger = KotlinLogging.logger {}

class AsgRelationshipTransformer(declarationCache: DeclarationCache) : BaseTransformer(declarationCache) {

    companion object {
        private var ctIterable: CtTypeReference<Iterable<*>>? = null
    }

    fun transform(ctTypeReference: CtTypeReference<*>, modelComponents: ModelComponents) {
        if (ctIterable == null) {
            ctIterable = ctTypeReference.factory.Type().get<Iterable<*>>(Iterable::class.java).reference
        }

        val stopWatch = Stopwatch.createStarted()

        val transformer = Transformer(ctTypeReference, modelComponents)
        transformer.scan(ctTypeReference)

        logger.debug { "Transformed ${ctTypeReference.canonicalName()} ($stopWatch)" }
    }


    private inner class Transformer(private var thisCtType: CtTypeReference<*>,
                                    modelComponents: ModelComponents) : SpoonVisitor() {

        private val types: MutableMap<String, Type> by modelComponents.components
        private val typeParametersMappings: MutableMap<String, TypeParameterMapping> by modelComponents.components
        private val properties: MutableMap<String, Property> by modelComponents.components
        private val executables: MutableMap<String, Executable> by modelComponents.components
        private val parameters: MutableMap<String, Parameter> by modelComponents.components

        private val invokes: MutableMap<Pair<Executable, Executable>, Invocation> = mutableMapOf()
        private val accesses: MutableMap<Pair<Executable, Property>, Access> = mutableMapOf()
        private val elementTypes: MutableMap<CtTypeReference<*>, ElementType> = mutableMapOf()

        private var thisType: Type

        private var currentExecutable: Executable? = null
        private var currentCtExecutable: CtExecutable<*>? = null
        private var currentCtExecutableReference: CtExecutableReference<*>? = null
        private var currentCtInterfaceExecutables = mutableSetOf<CtExecutableReference<*>>()

        init {
            thisType = types[thisCtType.canonicalName()]!!
        }

        private val visitedElements = mutableSetOf<String>()

        override fun scan(element: CtElement?) {
            if (element == null) return

            val elementName = element.canonicalName()
            if (!visitedElements.contains(elementName) ||
                    element is CtFieldReference<*> ||
                    element is CtInvocation<*> ||
                    element is CtFieldAccess<*>) {
                logger.trace { "Visiting: $elementName" }
                visitedElements.add(elementName)
                super.scan(element)
            }
        }

        override fun <T : Any?> visitCtTypeReference(r: CtTypeReference<T>) {
            assert(r == thisCtType)

            types[r.canonicalName()]?.let { type ->
                r.superclass?.let { types[it.canonicalName()] }
                        ?.let { type.isExtending.add(it) }

                r.safeDeclaration().ifPresent {
                    r.superInterfaces.mapNotNull { types[it.canonicalName()] }
                            .let { type.isExtending.addAll(it) }

                    currentCtInterfaceExecutables.clear()
                    currentCtInterfaceExecutables.addAll(it.superInterfaces
                            .filter { it.safeDeclaration().isPresent }
                            .flatMap { it.allExecutables })

                    // FIXME: Needed because of a bug in spoon
                    try {
                        scan(it.allFields)
                    } catch (e: Exception) {
                        logger.error(e) { "Could not load all fields as an exception was encountered. Scanning only declared fields." }
                        scan(it.fields)
                    }
                    try {
                        scan(it.allExecutables)
                    } catch (e: Exception) {
                        scan(it.declaredExecutables)
                        logger.error(e) { "Could not load all fields as an exception was encountered. Scanning only declared executables." }
                    }
                    scan(it.formalCtTypeParameters)
                }
            }
        }

        override fun visitCtTypeParameter(t: CtTypeParameter) {

            types[t.reference.canonicalName()]?.also { parameter ->
                types[t.reference.boundingType.canonicalName()]?.let {
                    parameter.isExtending.add(it)
                }

                val declarer = t.typeParameterDeclarer
                when (declarer) {
                    is CtType<*> -> types[declarer.reference.canonicalName()]?.typeParameters?.add(parameter)
                    is CtMethod<*> -> executables[declarer.reference.canonicalName()]?.typeParameters?.add(parameter)
                }

            }
        }

        override fun <T : Any?> visitCtFieldReference(r: CtFieldReference<T>) {

            properties[r.qualifiedName]?.also { property ->
                createElementType(r.type)?.also { elementType ->
                    if (r.declaringType == thisCtType) {

                        property.type = elementType

                        thisType.properties.add(property)
                        thisType.inheritedProperties
                                .filter { it.name == property.name && !it.isStatic && !property.isStatic }
                                .let { property.shadows.addAll(it) }

                    } else if (r.declaringType.canonicalName() in types && property.visibility != Visibility.PRIVATE && !property.isStatic) {

                        thisType.inheritedProperties.add(property)
                        thisType.properties
                                .find { it.name == property.name && !it.isStatic }
                                ?.shadows
                                ?.add(property)

                    }
                }
            }
        }

        override fun <T : Any?> visitCtExecutableReference(r: CtExecutableReference<T>) {

            val declarationOptional = r.safeDeclaration()

            executables[r.canonicalName()]?.let { executable ->
                if (r.declaringType == thisCtType) {

                    thisType.executables.add(executable)

                    createElementType(r.type)?.let { executable.type = it }


                    declarationOptional.ifPresent {
                        currentCtExecutable = it
                        currentCtExecutableReference = r
                        currentExecutable = executable

                        visitOverridingExecutable()
                        scan(it.parameters)
                        scan(it.getElements({
                            it is CtFieldAccess<*> ||
                                    it is CtConstructorCall<*> ||
                                    it is CtInvocation<*> ||
                                    it is CtReturn<*>
                        }))
                    }
                } else if (r.declaringType.canonicalName() in types &&
                        executable.visibility != Visibility.PRIVATE &&
                        !executable.isStatic) {
                    thisType.inheritedExecutables.add(executable)
                }
            }

            declarationOptional.ifPresent {
                if (it is CtMethod<*> && it.formalCtTypeParameters.isNotEmpty()) {
                    scan(it.formalCtTypeParameters)
                }
            }
        }

        private fun visitOverridingExecutable() {
            currentCtInterfaceExecutables
                    .filter { currentCtExecutableReference!!.isOverriding(it) }
                    .mapNotNull { executables[it.canonicalName()] }
                    .forEach {
                        val canonicalName = Overriding.canonicalNameOf(currentExecutable!!, it, OverridingQuality.IMPLEMENTING)
                        currentExecutable!!.overrides.add(Overriding(canonicalName.sha256().toString(), canonicalName, currentExecutable!!, it, OverridingQuality.IMPLEMENTING))
                    }

            currentCtExecutableReference!!.overridingExecutable
                    ?.let { executables[it.canonicalName()] }
                    ?.let { superExec ->
                        if (superExec.isAbstract) {
                            val canonicalName = Overriding.canonicalNameOf(currentExecutable!!, superExec, OverridingQuality.IMPLEMENTING)
                            currentExecutable!!.overrides.add(Overriding(canonicalName.sha256().toString(), canonicalName, currentExecutable!!, superExec, OverridingQuality.IMPLEMENTING))
                        } else {
                            val superCalls = currentCtExecutable!!.getElements<CtInvocation<*>>({
                                it is CtInvocation<*> && it.target is CtSuperAccess<*>
                            })

                            val revertingCall = superCalls.find { it.executable.canonicalName() != superExec.canonicalName } != null
                            val overridingCall = superCalls.find { it.executable.canonicalName() == superExec.canonicalName } != null

                            val statementCount = currentCtExecutable!!.body?.statements?.size
                            val isReplacing = when {
                                statementCount == null -> OverridingQuality.REDECLARING
                                !overridingCall && statementCount == 0 -> OverridingQuality.CLEARING
                                revertingCall && statementCount == 1 -> OverridingQuality.REVERTING
                                overridingCall && statementCount == 1 -> OverridingQuality.DEFERRING
                                overridingCall && statementCount > 1 -> OverridingQuality.EXTENDING
                                else -> OverridingQuality.REPLACING
                            }
                            val canonicalName = Overriding.canonicalNameOf(currentExecutable!!, superExec, isReplacing)
                            currentExecutable!!.overrides.add(Overriding(canonicalName.sha256().toString(), canonicalName, currentExecutable!!, superExec, isReplacing))
                        }
                    }

        }

        override fun <T : Any?> visitCtParameter(p: CtParameter<T>) {

            val parameterId = Parameter.canonicalNameOf(
                    currentExecutable!!,
                    currentCtExecutable!!.parameters.indexOf(p)
            )

            parameters[parameterId]?.let { parameter ->

                currentExecutable!!.parameters.add(parameter)

                createElementType(p.type)?.let { parameter.type = it }
            }
        }

        override fun <T : Any?> visitCtFieldRead(e: CtFieldRead<T>) {
            visitCtFieldAccess(e, visitDelegateExpression(e.target), AccessType.READ)
        }

        override fun <T : Any?> visitCtFieldWrite(e: CtFieldWrite<T>) {
            visitCtFieldAccess(e, visitDelegateExpression(e.target), AccessType.WRITE)
        }

        private fun visitCtFieldAccess(fieldAccess: CtFieldAccess<*>, target: at.jku.isse.gradient.model.CanonicalEntity?, accessType: AccessType) {
            fieldAccess.variable.safeDeclaration().ifPresent {
                properties[it.reference.qualifiedName]?.also { callee ->
                    val isIterative = fieldAccess.getParent(TypeFilter(CtLoop::class.java)) != null

                    val cardinality = if (isIterative) Cardinality.UNBOUND else Cardinality.ONE

                    var access = accesses[Pair(currentExecutable!!, callee)]

                    if (access == null) {
                        val accessName = Access.canonicalNameOf(currentExecutable!!, target, callee, cardinality, accessType)
                        access = Access(accessName.sha256().toString(), accessName, currentExecutable!!, target, callee, cardinality, accessType)

                        currentExecutable!!.accesses.add(access)

                        accesses.put(Pair(currentExecutable!!, callee), access)
                        visitAccessor(access)
                    } else {

                        val newAccessType = if (access.accessType != accessType) AccessType.FULL else accessType
                        val newCardinality = when {
                            isIterative -> Cardinality.UNBOUND
                            access.cardinality < Cardinality.N -> Cardinality.N
                            else -> access.cardinality
                        }
                        val accessName = Access.canonicalNameOf(currentExecutable!!, target, callee, newCardinality, newAccessType)
                        val newAccess = Access(accessName.sha256().toString(), accessName, currentExecutable!!, target, callee, newCardinality, newAccessType)

                        currentExecutable!!.accesses.remove(access)
                        currentExecutable!!.accesses.add(newAccess)

                        accesses.put(Pair(currentExecutable!!, callee), newAccess)
                    }
                }
            }
        }

        private fun visitAccessor(access: Access) {
            val executableName = access.source.name.toLowerCase()
            if ("get" + access.target.name.toLowerCase() in executableName
                    && access.accessType == AccessType.READ) {
                currentExecutable!!.accessor = Accessor.GETTER
            } else if ("set" + access.target.name.toLowerCase() in executableName
                    && access.accessType == AccessType.WRITE) {
                currentExecutable!!.accessor = Accessor.SETTER
            }
        }

        override fun <T : Any?> visitCtConstructorCall(e: CtConstructorCall<T>) {
            visitCtAbstractInvocation(e, visitDelegateExpression(e.target))
        }

        override fun <T : Any?> visitCtInvocation(e: CtInvocation<T>) {
            visitCtAbstractInvocation(e, visitDelegateExpression(e.target))
        }

        private fun <T : Any?> visitDelegateExpression(expression: CtExpression<T>?): at.jku.isse.gradient.model.CanonicalEntity? {
            return when {
                expression is CtFieldAccess<*> && expression.variable.declaringType != null -> {
                    properties[expression.variable.qualifiedName]
                }
                expression is CtArrayReadImpl<*> -> {
                    val target = expression.target
                    when {
                        target is CtFieldAccess<*> && target.variable.declaringType != null -> {
                            properties[target.variable.qualifiedName]
                        }
                        else -> {
                            null
                        }
                    }
                }
                expression is CtVariableAccess<*> -> {
                    val variable = expression.variable
                    if (variable is CtParameterReference<*>) {
                        val declarationOptional = variable.safeDeclaration()
                        if (declarationOptional.isPresent) {
                            val parameterName = Parameter.canonicalNameOf(currentExecutable!!,
                                    currentCtExecutable!!.parameters.indexOf(declarationOptional.get()))
                            parameters[parameterName]
                        } else {
                            null
                        }
                    } else {
                        null
                    }
                }
                expression is CtTypeAccess<*> -> {
                    types[expression.accessedType.canonicalName()]
                }
                else -> {
                    null
                }
            }
        }

        private fun <T : Any?> visitCtAbstractInvocation(ctInvocation: CtAbstractInvocation<T>, delegate: at.jku.isse.gradient.model.CanonicalEntity?) {
            ctInvocation.executable.safeDeclaration().ifPresent {
                executables[it.reference.canonicalName()]?.also { callee ->
                    val isIterative = ctInvocation.getParent(TypeFilter(CtLoop::class.java)) != null

                    val cardinality = if (isIterative) Cardinality.UNBOUND else Cardinality.ONE

                    var invoke = invokes[Pair(currentExecutable!!, callee)]


                    if (invoke == null) {
                        val invocationName = Invocation.canonicalNameOf(currentExecutable!!, delegate, callee, cardinality)
                        invoke = Invocation(invocationName.sha256().toString(), invocationName, currentExecutable!!, delegate, callee, cardinality)

                        currentExecutable!!.invokes.add(invoke)

                        invokes.put(Pair(currentExecutable!!, callee), invoke)

                    } else if (invoke.cardinality != Cardinality.UNBOUND) {

                        val newCardinality = when {
                            isIterative -> Cardinality.UNBOUND
                            invoke.cardinality < Cardinality.N -> Cardinality.N
                            else -> invoke.cardinality
                        }

                        val invocationName = Invocation.canonicalNameOf(currentExecutable!!, delegate, callee, newCardinality)
                        val newInvoke = Invocation(invocationName.sha256().toString(), invocationName, currentExecutable!!, delegate, callee, newCardinality)

                        currentExecutable!!.invokes.remove(invoke)
                        currentExecutable!!.invokes.add(newInvoke)

                        invokes.put(Pair(currentExecutable!!, callee), newInvoke)
                    }
                }
            }
        }

        override fun <T : Any?> visitCtReturn(returnStatement: CtReturn<T>) {
            returnStatement.returnedExpression?.type
                    ?.let {
                        createElementType(it)
                                ?.also { currentExecutable!!.returns.add(it) }
                    }
        }

        private fun createElementType(r: CtTypeReference<*>): ElementType? {

            var elementType: ElementType? = elementTypes[r]

            if (elementType == null) {
                if (r !is CtArrayTypeReference<*> &&
                        !r.isGenerics &&
                        r.safeDeclaration().isPresent && // needed for subtype of
                        r.isSubtypeOf(ctIterable) &&
                        r.actualTypeArguments.isNotEmpty()) {
                    val typeName = r.actualTypeArguments.first().canonicalName()

                    types[typeName]?.let { type ->
                        val cardinality = Cardinality.UNBOUND

                        val canonicalName = ElementType.canonicalNameOf(type, cardinality, emptyList())

                        elementType = ElementType(canonicalName.sha256().toString(), canonicalName, type, emptyList(), cardinality)
                    }

                } else {
                    types[r.canonicalName()]?.let { baseType ->
                        val cardinality = when (r) {
                            is CtArrayTypeReference<*> -> Cardinality.N
                            else -> Cardinality.ONE
                        }

                        val typeParameters = createElementTypeParameters(r)

                        val canonicalName = ElementType.canonicalNameOf(baseType, cardinality, typeParameters)

                        elementType = ElementType(canonicalName.sha256().toString(), canonicalName, baseType, typeParameters, cardinality)
                    }
                }
                elementType?.let { elementTypes.put(r, it) }
            }


            return elementType
        }

        private fun createElementTypeParameters(r: CtTypeReference<*>): List<TypeParameterMapping> {
            return r.actualTypeArguments.mapNotNull { actualCtType ->
                val declarationOptional = actualCtType.safeDeclaration()
                if (declarationOptional.isPresent) {
                    types[declarationOptional.get().reference.canonicalName()]?.let { parameter ->

                        types[actualCtType.canonicalName()]?.let { actualType ->
                            val canonicalName = TypeParameterMapping.canonicalNameOf(parameter, actualType)

                            typeParametersMappings.getOrPut(canonicalName) {
                                TypeParameterMapping(
                                        id = canonicalName.sha256().toString(),
                                        canonicalName = canonicalName,
                                        parameter = parameter,
                                        actualType = actualType
                                )
                            }
                        }
                    }
                } else {
                    null
                }
            }
        }
    }
}

