package at.jku.isse.gradient.dal.neo4j

import at.jku.isse.gradient.dal.GenericDao
import at.jku.isse.gradient.dal.StructuralDao
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.model.Version
import com.google.common.base.Preconditions.checkArgument
import com.google.inject.Inject
import mu.KotlinLogging
import java.util.*

private val logger = KotlinLogging.logger {}

open class StructuralDao
@Inject constructor(private val dao: GenericDao<at.jku.isse.gradient.model.Entity>) : StructuralDao {

    override fun appendNewVersion(projectId: String, version: Version) {
        checkArgument(projectId.isNotBlank())
        
        dao.createOrUpdate(version)

        appendNewLastRevision(projectId, version)
    }

    private fun appendNewLastRevision(projectId: String, version: Version) {
        val lastRevisionId = getLastRevisionId(projectId)
        if (lastRevisionId != null) {
            dao.query(
                    """
                    MATCH (project:Project {id:{projectId}}) -[revision:lastRevision]-> (oldVersion:Version {id:{oldVersionId}})
                    MATCH  (newVersion:Version {id:{newVersionId}})
                    WITH project, oldVersion, newVersion, revision
                    CREATE (project) -[:contains]-> (newVersion)
                    CREATE (oldVersion) -[:next]-> (newVersion)
                    CREATE (project) -[:lastRevision]-> (newVersion)
                    DELETE revision
                    """.trimIndent(),
                    mapOf(
                            Pair("projectId", projectId),
                            Pair("oldVersionId", lastRevisionId),
                            Pair("newVersionId", version.id)
                    )
            )
        } else {
            dao.query(
                    """
                    MATCH (project:Project {id:{projectId}})
                    MATCH (version:Version {id:{versionId}})
                    CREATE (project) -[:contains]-> (version)
                    CREATE (project) -[:lastRevision]-> (version)
                    """.trimIndent(),
                    mapOf(
                            Pair("projectId", projectId),
                            Pair("versionId", version.id)
                    )
            )
        }
    }

    override fun getOrPutProject(projectName: String, canonicalProjectName: String): String {

        val result = dao.query(
                "MATCH (project:Project {canonicalName:{canonicalName}}) RETURN project.id",
                mapOf(Pair("canonicalName", canonicalProjectName)),
                true
        )

        return if (result.isNotEmpty()) {
            result.first()["project.id"] as String
        } else {
            val project = Project(UUID.randomUUID().toString(), projectName, canonicalProjectName, emptyList())
            dao.createOrUpdate(project)
            project.id
        }
    }

    override fun getLastRevisionId(projectId: String): String? {
        checkArgument(projectId.isNotBlank())

        val result = dao.query(
                "MATCH (:Project {id:{id}}) -[:lastRevision]-> (v:Version) RETURN v.id",
                mapOf(Pair("id", projectId)),
                true
        )

        return if (result.isNotEmpty()) result.first()["v.id"] as String else null
    }

}