package at.jku.isse.gradient.dal.neo4j

import at.jku.isse.gradient.ProfilerState
import at.jku.isse.gradient.dal.DatabaseManager
import at.jku.isse.gradient.dal.GenericDao
import at.jku.isse.gradient.dal.QueryDao
import at.jku.isse.gradient.profiledDebug
import com.google.common.base.Stopwatch
import com.google.inject.Inject
import mu.KotlinLogging
import org.neo4j.driver.v1.Record
import org.neo4j.driver.v1.Transaction
import org.neo4j.driver.v1.Values

private val logger = KotlinLogging.logger {}


open class GenericDao<in T : at.jku.isse.gradient.model.Entity>
@Inject constructor(private val databaseManager: DatabaseManager,
                    private val nodeMapper: QueryNodeMapper,
                    private val relationshipMapper: QueryRelationshipMapper) : GenericDao<T>, QueryDao {


    override fun delete(entity: T) {
        logger.debug { "Deleting ${entity.id}" }

        delete(listOf(entity))
    }

    override fun delete(entities: Collection<T>) {
        logger.debug { "Deleting ${entities.size} entities" }

        databaseManager.neo4jSession()?.use { session ->
            session.writeTransaction { tx ->
                entities.forEach {
                    tx.run("DELETE DETACH ({id:{id}})", Values.parameters("id", it.id))
                }
                tx.success()
            }
        }
    }

    override fun createOrUpdate(entity: T) {
        logger.debug { "Creating or updating ${entity.id}" }

        createOrUpdate(listOf(entity))
    }

    override fun createOrUpdate(entities: Collection<T>) {
        logger.debug { "Creating or updating ${entities.size} entities" }

        val nodeQueries = nodeMapper.map(entities)
        val relationshipQueries = relationshipMapper.map(entities)

        logger.debug { "Issuing ${nodeQueries.size() + relationshipQueries.size()} queries" }
        logger.debug { "Issuing ${nodeQueries.size()} node queries" }

        databaseManager.neo4jSession()?.use {
            val stopwatch = Stopwatch.createStarted()

            it.writeTransaction { tx ->
                nodeQueries.entries()
                        .forEach { tx.run(it.key, mapOf("parameters" to it.value)) }
                tx.success()
            }

            logger.debug { "\tFinished after $stopwatch" }

            stopwatch.reset().start()

            logger.debug { "Issuing ${relationshipQueries.size()} relationship queries" }
            it.writeTransaction { tx ->
                relationshipQueries.entries()
                        .forEach { tx.run(it.key, mapOf("parameters" to it.value)) }
                tx.success()
            }
            logger.debug { "\tFinished after $stopwatch" }
        }
    }

    override fun query(query: String, parameters: Map<String, Any>, readOnly: Boolean): List<Map<String, Any>> {
        if (query.isBlank()) return emptyList()
        logger.profiledDebug { "Issuing query [readonly=$readOnly]" }

        var result: List<Record> = emptyList()

        val unit = { tx: Transaction ->
            val statementResult = tx.run(query, parameters)
            when {
                statementResult != null -> statementResult
                        .list()
                        .also { tx.success() }
                else -> emptyList()
            }
        }

        databaseManager.neo4jSession()?.use { session ->
            result = if (readOnly) {
                session.readTransaction(unit)
            } else {
                session.writeTransaction(unit)
            }
        }

        logger.profiledDebug(ProfilerState.STOP) { "Finished persisting" }
        return result.map { it.asMap() }
    }
}
