package at.jku.isse.gradient.monitoring;

import at.jku.isse.gradient.GradientConfig;
import at.jku.isse.gradient.GradientContextKt;
import at.jku.isse.gradient.dal.DalModule;
import at.jku.isse.gradient.service.BehavioralService;
import at.jku.isse.gradient.model.Observation;
import at.jku.isse.gradient.model.__Gradient_Observable__;
import com.fasterxml.uuid.Generators;
import com.fasterxml.uuid.impl.TimeBasedGenerator;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.primitives.Primitives;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import org.aspectj.lang.Signature;
import org.aspectj.lang.reflect.CodeSignature;

import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

public aspect Monitor {

    private final ThreadLocal<String> processId = new ThreadLocal<String>() {
        @Override
        protected String initialValue() {
            return uuid.generate().toString();
        }
    };

    private final TimeBasedGenerator uuid = Generators.timeBasedGenerator();

    @Inject
    private GradientConfig gradientConfig;

    @Inject
    @Named("runtimeId")
    private String runtimeId;

    @Inject
    private StructuralCache structuralCache;

    @Inject
    private BehavioralService behavioralService;

    @Inject
    @Named("dal.cleanup.bus")
    private EventBus eventBus;

    private final List<Map<String, Object>> buffer;
    private List<String> executableContext = new ArrayList<>();


    Monitor() {
        GradientContextKt.getInjector().injectMembers(this);
        eventBus.register(this);

        buffer = new ArrayList<>(gradientConfig.observationCacheSize());
    }


    pointcut fieldWrite(__Gradient_Observable__ contextObject, Object newState):
            target(contextObject) &&
                    set(* *) &&
                    args(newState);

    pointcut fieldRead(__Gradient_Observable__ contextObject):
            target(contextObject) &&
                    get(* *);

    pointcut construction(__Gradient_Observable__ contextObject):
            target(contextObject) &&
                    !initialization(at.jku.isse.gradient.model.__Gradient_Observable__.new()) &&
                    initialization(*.new(..));

    pointcut invocation(__Gradient_Observable__ contextObject):
            target(contextObject) &&
                    !initialization(at.jku.isse.gradient.model.__Gradient_Observable__.new()) &&
                    !execution(java.lang.String at.jku.isse.gradient.model.__Gradient_Observable__.__gradient_id__()) &&
                    execution(* *.*(..));


    private ImmutableMap<String, Object> parseObservationValue(String elementName, Object newState) {
        if (newState instanceof Number) {
            return ImmutableMap.of("elementName", elementName, "value", newState, "type", "NUMBER");
        } else if (newState instanceof String) {
            return ImmutableMap.of("elementName", elementName, "value", newState, "type", "TEXT");
        } else if (newState instanceof __Gradient_Observable__) {
            return ImmutableMap.of("elementName", elementName, "value",
                    ((__Gradient_Observable__) newState).__gradient_id__(), "type", "REFERENCE");
        } else {
            if (newState == null) {
                return ImmutableMap.of();
            } else {
                return ImmutableMap.of("elementName", elementName, "value", newState.hashCode(), "type", "UNKNOWN");
            }
        }
    }


    after (__Gradient_Observable__ contextObject, Object newState): fieldWrite(contextObject, newState){
        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", uuid.generate().toString())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.WRITE.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableList<ImmutableMap<String, Object>> values = ImmutableList.of(parseObservationValue(elementName, newState));
        if (!values.isEmpty()) {
            observation.put("writes", values);
        }

        reportObservation(observation.build());
    }

    after (__Gradient_Observable__ contextObject) returning (Object field): fieldRead(contextObject){
        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", uuid.generate().toString())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.READ.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));
        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableList<ImmutableMap<String, Object>> values = ImmutableList.of(parseObservationValue(elementName, field));
        if (!values.isEmpty()) {
            observation.put("reads", values);
        }

        reportObservation(observation.build());
    }

    before(__Gradient_Observable__ contextObject): construction(contextObject){
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        executableContext.add(uuid.generate().toString());
        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();
        final Object[] arguments = thisJoinPoint.getArgs();

        final Map<String, Object> observation = beforeExecutable(contextObject, signature, arguments);

        reportObservation(observation);
    }


    after(__Gradient_Observable__ contextObject): construction(contextObject){
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();

        final Map<String, Object> observation = afterExecutable(contextObject, signature, null);

        executableContext.remove(executableContext.size() - 1);
        reportObservation(observation);
    }

    before(__Gradient_Observable__ contextObject): invocation(contextObject){
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        executableContext.add(uuid.generate().toString());
        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();
        final Object[] arguments = thisJoinPoint.getArgs();

        final Map<String, Object> observation = beforeExecutable(contextObject, signature, arguments);

        reportObservation(observation);
    }

    after(__Gradient_Observable__ contextObject) returning (Object returnValue): invocation(contextObject){
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();

        final Map<String, Object> observation = afterExecutable(contextObject, signature, returnValue);

        executableContext.remove(executableContext.size() - 1);
        reportObservation(observation);
    }

    public void reset() {
        buffer.clear();
        structuralCache.reload();
    }

    @Subscribe
    public void cleanupSignal(DalModule.Cleanup cleanup) {
        flush();
        behavioralService.reportObservationEnd(structuralCache.getProjectName(), structuralCache.getVersionId(), runtimeId);
    }

    public void flush() {
        behavioralService.report(structuralCache.getProjectName(), structuralCache.getVersionId(), runtimeId, buffer);
        buffer.clear();
    }

    private String canonicalExecutableName(CodeSignature signature) {
        final String declaringType = signature.getDeclaringTypeName();
        final String element = Objects.equals(signature.getName(), "<init>")
                ? signature.getDeclaringType().getSimpleName() : signature.getName();
        final String parameters = Arrays.<Class<?>>stream(signature.getParameterTypes())
                .map(Primitives::wrap)
                .map(Class::getTypeName)
                .collect(Collectors.joining(", "));

        return String.format("%s#%s(%s)", declaringType, element, parameters);
    }

    private void reportObservation(Map<String, Object> observation) {
        if (buffer.size() + 1 > gradientConfig.observationCacheSize()) {
            flush();
        }

        buffer.add(observation);
    }

    private Map<String, Object> beforeExecutable(__Gradient_Observable__ contextObject, CodeSignature signature, Object[] arguments) {
        final String[] parameters = signature.getParameterNames();
        final String elementName = canonicalExecutableName(signature);
        assert arguments.length == parameters.length;


        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", uuid.generate().toString())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.CALL.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final List<Map<String, Object>> values = new ArrayList<>(arguments.length);
        for (int i = 0; i < arguments.length; i++) {
            values.add(parseObservationValue(elementName + "#" + i, arguments[i]));
        }

        if (!values.isEmpty()) {
            observation.put("receives", values);
        }

        return observation.build();
    }

    private Map<String, Object> afterExecutable(__Gradient_Observable__ contextObject, CodeSignature signature, Object returnValue) {
        String elementName = canonicalExecutableName(signature);

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", uuid.generate().toString())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.RETURN.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableMap<String, Object> value = parseObservationValue(elementName, returnValue);
        if (!value.isEmpty()) {
            observation.put("returns", ImmutableList.of(value));
        }

        return observation.build();
    }
}
