package at.jku.isse.gradient.java

import at.jku.isse.gradient.model.GradientType
import com.google.common.hash.HashCode
import com.google.common.hash.Hashing
import spoon.reflect.code.CtFieldAccess
import spoon.reflect.code.CtInvocation
import spoon.reflect.declaration.*
import spoon.reflect.reference.*
import java.lang.NullPointerException
import java.util.*

internal fun CtTypeReference<*>.canonicalName(): String {
    return when (this) {
        is CtArrayTypeReference<*> -> this.componentType.canonicalName()
        is CtTypeParameterReference -> canonicalName()
        else -> box().qualifiedName
    }
}

internal fun CtExecutableReference<*>.canonicalName(): String {
    val declaringType = this.declaringType.canonicalName()
    val executableName = if (this.isConstructor) this.declaringType.simpleName else this.simpleName

    val parameters = this.parameters.joinToString(", ") { parameter ->
        if (parameter.isGenerics && parameter is CtTypeParameterReference) {
            parameter.canonicalName()
        } else {
            parameter.box().qualifiedName
        }
    }

    return "$declaringType#$executableName($parameters)"
}

internal fun CtTypeParameterReference.canonicalName(): String {
    val boundingType = if (this.boundingType == null) "" else ": ${this.boundingType.canonicalName()}"

    return "${this.qualifiedName} $boundingType"
}

internal fun CtElement.canonicalName(): String {
    return when (this) {
        is CtTypeReference<*> -> this.canonicalName()
        is CtExecutableReference<*> -> this.canonicalName()
        is CtTypeParameterReference -> this.canonicalName()
        is CtParameter<*> -> "${this.reference.declaringExecutable.canonicalName()} <- ${this.simpleName}"
        is CtFieldAccess<*> -> "${this.target} -> ${this.variable}"
        is CtInvocation<*> -> "${this.target} -> ${this.executable}"
        else -> this.toString()
    }
}

abstract class BaseTransformer(private val declarationCache: DeclarationCache) {

    companion object {
        private val hasher = Hashing.sha256()
        internal val emptyHash = hasher.hashString("", Charsets.UTF_8)
    }


    internal fun CtTypeReference<*>.safeDeclaration(): Optional<CtType<*>> {
        return declarationCache.type.get(this)
    }

    internal fun CtFieldReference<*>.safeDeclaration(): Optional<CtField<*>> {
        return declarationCache.property.get(this)
    }

    internal fun CtExecutableReference<*>.safeDeclaration(): Optional<CtExecutable<*>> {
        return declarationCache.executable.get(this)
    }

    internal fun CtParameterReference<*>.safeDeclaration(): Optional<CtParameter<*>> {
        return declarationCache.parameter.get(this)
    }

    internal fun CtElement.sha256(): HashCode {
        return try {
            hasher.hashString(toString(), Charsets.UTF_8)
        } catch (e: NullPointerException) {
            hasher.hashInt(hashCode())
        }
    }

    internal fun String.sha256(): HashCode {
        return hasher.hashString(this, Charsets.UTF_8)
    }

    internal fun CtTypeReference<*>.toGradientType(): GradientType {
        return when (this.canonicalName()) {
            "java.lang.String" -> GradientType.TEXT
            "java.lang.Boolean",
            "java.lang.Byte",
            "java.lang.Character",
            "java.lang.Short",
            "java.lang.Integer",
            "java.lang.Long",
            "java.lang.Float",
            "java.lang.Double"
            -> GradientType.NUMBER
            "java.lang.Void" -> GradientType.VOID
            else -> GradientType.REFERENCE

        }
    }
}