package at.jku.isse.gradient.java

import at.jku.isse.gradient.annotation.GradientModel
import at.jku.isse.gradient.model.*
import com.google.common.base.Stopwatch
import com.google.common.hash.Hashing
import mu.KotlinLogging
import spoon.reflect.declaration.*
import spoon.reflect.reference.CtExecutableReference
import spoon.reflect.reference.CtFieldReference
import spoon.reflect.reference.CtTypeReference

private val logger = KotlinLogging.logger {}

/**
 * Collects all entity, i.e., all explicitly modeled classes without the references among them.
 */
class AsgNodeTransformer(declarationCache: DeclarationCache) : BaseTransformer(declarationCache) {

    fun transform(ctTypeReference: CtTypeReference<*>): ModelComponents {

        val stopWatch = Stopwatch.createStarted()

        val transformer = Transformer(ctTypeReference)
        transformer.scan(ctTypeReference)

        logger.debug { "Transformed ${ctTypeReference.canonicalName()} ($stopWatch)" }

        return transformer.modelComponents
    }

    private inner class Transformer(val thisType: CtTypeReference<*>,
                                    val modelComponents: ModelComponents = ModelComponents()) : SpoonVisitor() {

        private val types: MutableMap<String, Type> by modelComponents.components
        private val properties: MutableMap<String, Property> by modelComponents.components
        private val executables: MutableMap<String, Executable> by modelComponents.components
        private val parameters: MutableMap<String, Parameter> by modelComponents.components

        private var currentExecutable: Executable? = null
        private var currentCtExecutable: CtExecutable<*>? = null

        private val visitedElements = mutableSetOf<String>()

        override fun scan(element: CtElement?) {
            if (element == null) return

            val elementName = element.canonicalName()
            if (!visitedElements.contains(elementName)) {
                logger.trace { "Visiting: $elementName" }
                visitedElements.add(elementName)
                super.scan(element)
            }
        }

        override fun <T : Any?> visitCtTypeReference(r: CtTypeReference<T>) {
            assert(r == thisType)

            val typeOptional = r.safeDeclaration()
            val versionHash = if (typeOptional.isPresent) typeOptional.get().sha256() else emptyHash

            val canonicalName = r.canonicalName()

            val type = Type(
                    id = Hashing.combineOrdered(listOf(canonicalName.sha256(), versionHash)).toString(),
                    name = r.simpleName,
                    canonicalName = canonicalName,
                    availability = Availability.UNAVAILABLE,
                    quality = Quality.CONCRETE,
                    visibility = Visibility.PUBLIC,
                    isGenerics = false,
                    isGradientModel = false,
                    versionHash = versionHash
            )
            types.put(type.canonicalName, type)

            typeOptional.ifPresent {
                type.run {
                    availability = if (r.declaration != null) Availability.SOURCE else Availability.BINARY
                    quality = when {
                        r.isInterface -> Quality.PURE
                        ModifierKind.ABSTRACT in r.modifiers -> Quality.PARTIAL
                        else -> Quality.CONCRETE
                    }
                    visibility = when {
                        ModifierKind.PRIVATE in r.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in r.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    }
                    isGradientModel = it.getAnnotation(GradientModel::class.java) != null
                }
                scan(it.allFields)
                scan(it.allExecutables)
                scan(it.formalCtTypeParameters)
            }
        }

        override fun visitCtTypeParameter(t: CtTypeParameter) {

            val versionHash = t.sha256()
            val canonicalName = t.reference.canonicalName()

            val typeParameter = Type(
                    id = Hashing.combineOrdered(listOf(canonicalName.sha256(), versionHash)).toString(),
                    name = t.simpleName,
                    canonicalName = canonicalName,
                    availability = Availability.UNAVAILABLE,
                    quality = Quality.PURE,
                    visibility = when {
                        ModifierKind.PRIVATE in t.typeParameterDeclarer.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in t.typeParameterDeclarer.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    },
                    isGenerics = true,
                    isGradientModel = false,
                    versionHash = versionHash
            )

            types.put(typeParameter.canonicalName, typeParameter)
        }

        override fun <T : Any?> visitCtFieldReference(r: CtFieldReference<T>) {
            val declarationOptional = r.safeDeclaration()

            val versionHash = if (declarationOptional.isPresent) declarationOptional.get().sha256() else emptyHash
            val qualifiedName = r.qualifiedName
            val property = Property(
                    id = Hashing.combineOrdered(listOf(qualifiedName.sha256(), versionHash)).toString(),
                    name = r.simpleName,
                    canonicalName = qualifiedName,
                    visibility = when {
                        ModifierKind.PRIVATE in r.modifiers -> Visibility.PRIVATE
                        ModifierKind.PROTECTED in r.modifiers -> Visibility.INHERITANCE
                        else -> Visibility.PUBLIC
                    },
                    isStatic = ModifierKind.STATIC in r.modifiers,
                    isImmutable = ModifierKind.FINAL in r.modifiers,
                    gradientType = r.type.toGradientType(),
                    versionHash = versionHash
            )

            properties.put(property.canonicalName, property)
        }

        override fun <T : Any?> visitCtExecutableReference(r: CtExecutableReference<T>) {

            val declarationOptional = r.safeDeclaration()

            declarationOptional.ifPresent { declaration ->
                if (declaration is CtTypeMember) {
                    val versionHash = declaration.sha256()
                    val canonicalName = r.canonicalName()
                    val executable = Executable(
                            id = Hashing.combineOrdered(listOf(canonicalName.sha256(), versionHash)).toString(),
                            name = r.simpleName,
                            canonicalName = canonicalName,
                            isAbstract = ModifierKind.ABSTRACT in declaration.modifiers || thisType.isInterface,
                            visibility = when {
                                ModifierKind.PRIVATE in declaration.modifiers -> Visibility.PRIVATE
                                ModifierKind.PROTECTED in declaration.modifiers -> Visibility.INHERITANCE
                                else -> Visibility.PUBLIC
                            },
                            isStatic = ModifierKind.STATIC in declaration.modifiers,
                            isConstructor = r.isConstructor,
                            gradientType = r.type.toGradientType(),
                            versionHash = versionHash
                    )

                    executables.put(executable.canonicalName, executable)

                    currentCtExecutable = declaration
                    currentExecutable = executable

                    if (declaration is CtMethod<*> && declaration.formalCtTypeParameters.isNotEmpty()) {
                        scan(declaration.formalCtTypeParameters)
                    }

                    scan(declaration.parameters)
                }
            }
        }

        override fun <T : Any?> visitCtParameter(p: CtParameter<T>) {

            val canonicalName = Parameter.canonicalNameOf(
                    executable = currentExecutable!!,
                    index = currentCtExecutable!!.parameters.indexOf(p)
            )

            val parameter = Parameter(
                    id = canonicalName.sha256().toString(),
                    name = p.simpleName,
                    canonicalName = canonicalName,
                    gradientType = p.type.toGradientType()
            )

            parameters.put(parameter.canonicalName, parameter)
        }
    }
}