package at.jku.isse.gradient.dal.neo4j

import at.jku.isse.gradient.GradientConfig
import mu.KotlinLogging
import org.aeonbits.owner.ConfigFactory
import org.neo4j.driver.internal.logging.ConsoleLogging
import org.neo4j.driver.v1.AuthTokens
import org.neo4j.driver.v1.Config
import org.neo4j.driver.v1.Driver
import org.neo4j.driver.v1.GraphDatabase
import java.util.concurrent.TimeUnit
import java.util.logging.Level


private val logger = KotlinLogging.logger {}


internal object Neo4jConfiguration {

    fun configure(): Pair<Driver, Runnable> {
        val config = ConfigFactory.create(GradientConfig::class.java)
        val driverConfig = Config.build()
                .withMaxTransactionRetryTime(30, TimeUnit.SECONDS)
                .withConnectionTimeout(1, TimeUnit.SECONDS)
                .withLogging(ConsoleLogging(Level.parse(config.neo4jLogLevel())))
                .toConfig()

        val connectionString = System.getenv("NEO4J_CONNECTION_STRING") ?: config.neo4jConnectionString()

        val driver = GraphDatabase.driver(
                connectionString,
                AuthTokens.basic(config.neo4jUsername(), config.neo4jPassword()),
                driverConfig
        )

        createIndexes(driver)

        val cleanup = Runnable {
            driver.close()
        }

        return Pair(driver, cleanup)
    }

    private fun createIndexes(driver: Driver) {
        logger.debug { "Creating indexes" }
        driver.session().use { session ->
            session.beginTransaction().use { tx ->
                tx.run("CREATE INDEX ON : Project(canonicalName)")

                tx.run("CREATE INDEX ON : Version(id)")

                tx.run("CREATE INDEX ON : Type(canonicalName)")
                tx.run("CREATE INDEX ON : Type(availability)")
                tx.run("CREATE INDEX ON : Type(isGradientModel)")

                tx.run("CREATE INDEX ON : Property(canonicalName)")

                tx.run("CREATE INDEX ON : Executable(canonicalName)")

                tx.run("CREATE INDEX ON : Invocation(canonicalName)")

                tx.run("CREATE INDEX ON : Parameter(canonicalName)")

                tx.run("CREATE INDEX ON : TypeParameter(canonicalName)")

                tx.run("CREATE INDEX ON : ElementType(canonicalName)")

                tx.success()
            }
        }
    }
}