package at.jku.isse.gradient.dal

import at.jku.isse.gradient.GradientConfig
import at.jku.isse.gradient.dal.mongo.BatchMongoObservationDao
import at.jku.isse.gradient.dal.mongo.MongoConfiguration
import at.jku.isse.gradient.dal.neo4j.Neo4jConfiguration
import at.jku.isse.gradient.dal.neo4j.QueryNodeMapper
import at.jku.isse.gradient.dal.neo4j.QueryRelationshipMapper
import at.jku.isse.gradient.model.Project
import at.jku.isse.gradient.model.Type
import at.jku.isse.psma.gradient.server.BehavioralServiceGrpc
import at.jku.isse.psma.gradient.server.StructuralServiceGrpc
import com.google.common.eventbus.EventBus
import com.google.inject.AbstractModule
import com.google.inject.Provides
import com.google.inject.Singleton
import com.google.inject.TypeLiteral
import com.google.inject.name.Names
import com.mongodb.client.MongoClient
import io.grpc.Channel
import io.grpc.ManagedChannelBuilder
import mu.KotlinLogging
import org.aeonbits.owner.ConfigFactory
import org.neo4j.driver.v1.Driver
import org.neo4j.driver.v1.Session
import java.io.File
import java.util.*

private val logger = KotlinLogging.logger {}

class DalModule : AbstractModule() {

    object Cleanup

    private val cleanupBus = EventBus("Dal Cleanup")
    private val cleanupProcedures = mutableListOf<Runnable>()

    override fun configure() {

        bindConfigurations()

        bindNeo4j()

        bindMongo()

        bindDaos()

        bindMappers()

        bind(EventBus::class.java)
                .annotatedWith(Names.named("dal.cleanup.bus"))
                .toInstance(cleanupBus)

        Runtime.getRuntime().addShutdownHook(Thread(Runnable {
            cleanupBus.post(Cleanup)
            cleanupProcedures.forEach { it.run() }
        }))
    }

    private fun bindMappers() {
        bind(LabelVisitor::class.java)
        bind(QueryNodeMapper::class.java)
        bind(QueryRelationshipMapper::class.java)
    }

    private fun bindNeo4j() {
        val (driver, cleanup) = Neo4jConfiguration.configure()
        bind(Driver::class.java)
                .toInstance(driver)

        this.cleanupProcedures.add(cleanup)
    }

    private fun bindMongo() {
        val runtimeId = UUID.randomUUID().toString()
        bind(String::class.java)
                .annotatedWith(Names.named("runtimeId"))
                .toInstance(runtimeId)

        val (client, cleanup) = MongoConfiguration.configure(runtimeId)
        bind(MongoClient::class.java)
                .toInstance(client)

        this.cleanupProcedures.add(cleanup)
    }

    private fun bindConfigurations() {
        val coreConfig = ConfigFactory.create(GradientConfig::class.java)
        bind(GradientConfig::class.java)
                .toInstance(coreConfig)

        val gradientFolder = coreConfig.gradientFolder()
        gradientFolder.mkdir()
        bind(File::class.java)
                .annotatedWith(Names.named("gradientFolder"))
                .toInstance(gradientFolder)

        val structuralProperties = File(gradientFolder, "structural.properties")
        structuralProperties.createNewFile()

        bind(File::class.java)
                .annotatedWith(Names.named("structural.properties"))
                .toInstance(structuralProperties)
    }

    private fun bindDaos() {
        bind(object : TypeLiteral<GenericDao<at.jku.isse.gradient.model.Entity>>() {})
                .to(object : TypeLiteral<at.jku.isse.gradient.dal.neo4j.GenericDao<at.jku.isse.gradient.model.Entity>>() {})

        bind(object : TypeLiteral<GenericDao<Project>>() {})
                .to(object : TypeLiteral<at.jku.isse.gradient.dal.neo4j.GenericDao<Project>>() {})

        bind(object : TypeLiteral<GenericDao<Type>>() {})
                .to(object : TypeLiteral<at.jku.isse.gradient.dal.neo4j.GenericDao<Type>>() {})

        bind(QueryDao::class.java)
                .to(at.jku.isse.gradient.dal.neo4j.GenericDao::class.java)

        bind(ObservationDao::class.java)
                .to(BatchMongoObservationDao::class.java)

        bind(StructuralDao::class.java)
                .to(at.jku.isse.gradient.dal.neo4j.StructuralDao::class.java)
    }


    @Provides
    fun provideStructuralServiceStub(channel: Channel): StructuralServiceGrpc.StructuralServiceStub {
        return StructuralServiceGrpc.newStub(channel)
    }

    @Provides
    fun provideBehavioralServiceStub(channel: Channel): BehavioralServiceGrpc.BehavioralServiceBlockingStub {
        return BehavioralServiceGrpc.newBlockingStub(channel)
    }

    @Provides
    @Singleton
    fun provideChannel(config: GradientConfig): Channel {
        return ManagedChannelBuilder
                .forAddress(config.gradientServerHost(), config.gradientServerPort())
                .usePlaintext()
                .build()
    }

    @Provides
    fun provideSession(driver: Driver): Session {
        return driver.session()
    }
}