package at.jku.isse.gradient.monitoring;

import at.jku.isse.gradient.Gradient;
import at.jku.isse.gradient.runtime.__Gradient_Observable__;
import at.jku.isse.gradient.service.EventService;
import com.google.common.primitives.Primitives;
import com.google.inject.Inject;
import com.google.inject.Injector;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.CodeSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

@Aspect
public class Monitor {

    private static final Logger logger = LoggerFactory.getLogger(Monitor.class);
    private static final UUID EMPTY_UUID = UUID.fromString("00000000-0000-0000-0000-000000000000");

    public boolean monitoringEnabled;

    @Inject
    private EventService eventService;

    public Monitor() {
        logger.debug("Configuring monitoring aspect");

        final Gradient gradient = Gradient.Companion.getDefaultInstance();
        final Injector injector = gradient.bootstrapGradient();
        injector.injectMembers(this);

        monitoringEnabled = eventService != null;
        if (!monitoringEnabled) logger.warn("No event service is provided thus disabling monitor.");
    }

    @Pointcut(value = "target(contextObject) && " +
            "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
            "set(* at.jku.isse.gradient.runtime.__Gradient_Observable__+.*) && args(obj)", argNames = "contextObject,obj")
    public void propertyWrite(__Gradient_Observable__ contextObject, Object obj) {
    }

    @Pointcut(value = "target(contextObject) && " +
            "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
            "get(* at.jku.isse.gradient.runtime.__Gradient_Observable__+.*)", argNames = "contextObject")
    public void propertyRead(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) && " +
                    "execution(static * at.jku.isse.gradient.runtime.__Gradient_Observable__+.*(..))")
    public void executableStatic() {
    }

    @Pointcut(value =
            "target(contextObject) &&" +
                    "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
                    "(execution(void at.jku.isse.gradient.runtime.__Gradient_Observable__+.*(..)) || " +
                    "   execution(at.jku.isse.gradient.runtime.__Gradient_Observable__+.new(..)))",
            argNames = "contextObject")
    public void executableVoid(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "target(contextObject) &&" +
                    "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
                    "!execution(void at.jku.isse.gradient.runtime.__Gradient_Observable__+.*(..)) &&" +
                    "execution(* at.jku.isse.gradient.runtime.__Gradient_Observable__+.*(..))",
            argNames = "contextObject")
    public void executableReturning(__Gradient_Observable__ contextObject) {
    }

    @After(value = "propertyWrite(contextObject, obj)", argNames = "thisJoinPointStaticPart,contextObject,obj")
    public void propertyWriteAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object obj) {
        if (!monitoringEnabled) return;
        assert thisJoinPointStaticPart != null && contextObject != null;

        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        try {
            eventService.reportPropertyWrite(elementName, contextObject.__gradient_id__(), obj);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ propertyWriteAdvice", elementName));
        }
    }

    @AfterReturning(pointcut = "propertyRead(contextObject)", returning = "obj", argNames = "thisJoinPointStaticPart,contextObject,obj")
    public void propertyReadAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object obj) {
        if (!monitoringEnabled) return;
        assert thisJoinPointStaticPart != null && contextObject != null;

        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        try {
            eventService.reportPropertyRead(elementName, contextObject.__gradient_id__(), obj);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ propertyReadAdvice", elementName));
        }
    }


    @Before(value = "executableStatic()", argNames = "thisJoinPoint,thisJoinPointStaticPart")
    public void executableStaticCallAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart) {
        if (!monitoringEnabled) return;
        assert thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());
        executableCallAdviceHandler(thisJoinPoint, elementName, EMPTY_UUID);
    }

    @Before(value = "executableReturning(contextObject) || executableVoid(contextObject)", argNames = "thisJoinPoint,thisJoinPointStaticPart,contextObject")
    public void executableCallAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        if (!monitoringEnabled) return;
        assert contextObject != null && thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());
        executableCallAdviceHandler(thisJoinPoint, elementName, contextObject.__gradient_id__());
    }

    private void executableCallAdviceHandler(JoinPoint thisJoinPoint, String elementName, UUID gradientId) {

        try {
            eventService.reportExecutableCall(elementName, gradientId);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableCallAdvice (call)", elementName));
        }

        try {
            final Object[] arguments = thisJoinPoint.getArgs();
            for (int i = 0; i < arguments.length; i++) {
                final String parameterName = elementName + "#" + i;
                eventService.reportExecutableParameter(parameterName, gradientId, arguments[i]);
            }
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableCallAdvice (parameters)", elementName));
        }
    }


    @AfterReturning(value = "executableStatic()", returning = "obj", argNames = "thisJoinPointStaticPart,obj")
    public void executableStaticReturningAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, Object obj) {
        if (!monitoringEnabled) return;
        assert thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());

        try {
            eventService.reportExecutableReturn(elementName, EMPTY_UUID, obj);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableReturnAdvice", elementName));
        }
    }

    @AfterReturning(value = "executableReturning(contextObject)", returning = "obj", argNames = "thisJoinPointStaticPart,contextObject,obj")
    public void executableReturnAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object obj) {
        if (!monitoringEnabled) return;
        assert contextObject != null && thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());

        try {
            eventService.reportExecutableReturn(elementName, contextObject.__gradient_id__(), obj);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableReturnAdvice", elementName));
        }
    }

    @AfterReturning(value = "executableVoid(contextObject)", argNames = "thisJoinPointStaticPart,contextObject")
    public void executableReturnVoidAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        if (!monitoringEnabled) return;
        assert contextObject != null && thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());

        try {
            eventService.reportExecutableReturn(elementName, contextObject.__gradient_id__(), null);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableReturnVoidAdvice", elementName));
        }
    }

    @AfterThrowing(value = "executableStatic()", throwing = "exception", argNames = "thisJoinPointStaticPart,exception")
    public void executableStaticThrowingAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, Throwable exception) {
        if (!monitoringEnabled) return;
        assert thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());

        try {
            eventService.reportExecutableException(elementName, EMPTY_UUID, exception);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableThrowingAdvice", elementName));
        }
    }

    @AfterThrowing(value = "executableReturning(contextObject) || executableVoid(contextObject)", throwing = "exception", argNames = "thisJoinPointStaticPart,contextObject,exception")
    public void executableThrowingAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Throwable exception) {
        if (!monitoringEnabled) return;
        assert contextObject != null && thisJoinPointStaticPart != null;

        final String elementName = canonicalExecutableName((CodeSignature) thisJoinPointStaticPart.getSignature());

        try {
            eventService.reportExecutableException(elementName, contextObject.__gradient_id__(), exception);
        } catch (Exception e) {
            logger.debug(String.format("Could not report event: %s @ executableThrowingAdvice", elementName));
        }
    }

    private String canonicalExecutableName(CodeSignature signature) {
        assert signature != null;

        final String declaringType = signature.getDeclaringTypeName();
        final String element = Objects.equals(signature.getName(), "<init>")
                ? signature.getDeclaringType().getSimpleName() : signature.getName();
        final String parameters = Arrays.<Class<?>>stream(signature.getParameterTypes())
                .map(Primitives::wrap)
                .map(Class::getTypeName)
                .collect(Collectors.joining(", "));

        return String.format("%s#%s(%s)", declaringType, element, parameters);
    }
}
