package at.jku.isse.gradient.monitoring;

import at.jku.isse.gradient.Gradient;
import at.jku.isse.gradient.GradientConfig;
import at.jku.isse.gradient.GradientEvents;
import at.jku.isse.gradient.Util;
import at.jku.isse.gradient.dal.mongo.ConventionsKt;
import at.jku.isse.gradient.model.EventType;
import at.jku.isse.gradient.model.GradientType;
import at.jku.isse.gradient.model.Observation;
import at.jku.isse.gradient.runtime.__Gradient_Observable__;
import at.jku.isse.gradient.service.BehavioralService;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.primitives.Primitives;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.name.Named;
import kotlin.Pair;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.CodeSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

@Aspect
public class Monitor {

    private static final Logger logger = LoggerFactory.getLogger(Monitor.class);

    private final ThreadLocal<UUID> processId = ThreadLocal.withInitial(Util.INSTANCE::uuid);

    public final Gradient gradient = MonitoringContext.gradient;

    @Inject
    private GradientConfig gradientConfig;

    @Inject
    @Named("runtimeId")
    private UUID runtimeId;

    @Inject
    private StructuralCache structuralCache;

    @Inject
    private BehavioralService behavioralService;

    private final List<Map<String, Object>> buffer;

    private Stack<UUID> executableContext = new Stack<>();
    private Stack<String> elementNameContext = new Stack<>();

    public Monitor() {
        logger.debug("Configuring monitoring aspect");
        final Injector injector = gradient.bootstrapGradient();
        injector.injectMembers(this);

        buffer = new ArrayList<>(gradientConfig.observationCacheSize() + 1);
        final EventBus gradientEventBus = injector.getInstance(EventBus.class);
        gradientEventBus.register(this);

        executableContext.push(Util.INSTANCE.uuid());
        elementNameContext.push(ConventionsKt.ELEMENT_UNKNOWN);
    }

    @Pointcut(value = "target(contextObject) && " +
            "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
            "set(* *) && args(newState)", argNames = "contextObject,newState")
    public void fieldWrite(__Gradient_Observable__ contextObject, Object newState) {
    }

    @Pointcut(value = "target(contextObject) && " +
            "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
            "get(* *)", argNames = "contextObject")
    public void fieldRead(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "target(contextObject) && " +
                    "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
                    "!initialization(at.jku.isse.gradient.runtime.__Gradient_Observable__.new()) &&" +
                    "initialization(*.new(..))",
            argNames = "contextObject")
    public void construction(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "target(contextObject) && " +
                    "!target(at.jku.isse.gradient.runtime.__TimeBased_Gradient_Observable__) &&" +
                    "execution(* *.*(..))",
            argNames = "contextObject")
    public void invocation(__Gradient_Observable__ contextObject) {
    }


    @After(value = "fieldWrite(contextObject, state)", argNames = "thisJoinPointStaticPart,contextObject,state")
    public void fieldWriteAdvice(JoinPoint.StaticPart thisJoinPointStaticPart,
                                 __Gradient_Observable__ contextObject, Object state) {
        assert isInitialized();
        fieldAdvice(thisJoinPointStaticPart, contextObject, state, EventType.WRITE);
    }

    @AfterReturning(pointcut = "fieldRead(contextObject)", returning = "state",
            argNames = "thisJoinPointStaticPart,contextObject,state")
    public void fieldReadAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object state) {
        assert isInitialized();
        fieldAdvice(thisJoinPointStaticPart, contextObject, state, EventType.READ);
    }

    @After(value = "construction(contextObject)", argNames = "thisJoinPoint,thisJoinPointStaticPart,contextObject")
    public void constructionAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        assert isInitialized();
        callAdvice(thisJoinPoint, thisJoinPointStaticPart, contextObject);
    }

    @Before(value = "invocation(contextObject)", argNames = "thisJoinPoint,thisJoinPointStaticPart,contextObject")
    public void invocationBeforeAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        assert isInitialized();
        callAdvice(thisJoinPoint, thisJoinPointStaticPart, contextObject);
    }

    @AfterReturning(value = "invocation(contextObject)", returning = "returnValue", argNames = "thisJoinPointStaticPart,contextObject,returnValue")
    public void invocationAfterAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object returnValue) {
        assert isInitialized();

        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();

        final Map<String, Object> observation = afterExecutable(contextObject, signature, returnValue);

        assert !executableContext.empty() : "The context stack is empty, did you pop somewhere else?";
        executableContext.pop();
        assert !elementNameContext.empty() : "The parent element name stack is empty, did you pop somewhere else?";
        elementNameContext.pop();

        reportObservation(observation);
    }

    @Subscribe
    public void cleanupSignal(GradientEvents.Cleanup cleanup) {
        assert isInitialized();

        flush();

        if (structuralCache.present()) {
            behavioralService.reportObservationEnd(
                    Objects.requireNonNull(structuralCache.getProjectName()),
                    Objects.requireNonNull(structuralCache.getVersionId()),
                    runtimeId
            );
        }
    }

    private boolean isInitialized() {
        return gradientConfig != null && runtimeId != null && structuralCache != null && behavioralService != null;
    }

    private void fieldAdvice(JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject, Object field, EventType eventType) {
        assert isInitialized();

        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        final Map<String, Object> event = createEvent();
        event.put("object_id", contextObject.__gradient_id__());
        event.put("event_type", eventType.name());
        event.put("element_name", elementName);

        final Pair<String, Object> value = normalizeValue(field);
        event.put("value_type", value.getFirst());
        event.put("value", value.getSecond());

        reportObservation(event);
    }

    private void callAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();
        final Object[] arguments = thisJoinPoint.getArgs();

        final String elementName = canonicalExecutableName(signature);
        executableContext.push(Util.INSTANCE.uuid());
        elementNameContext.push(elementName);

        final List<Map<String, Object>> observation = beforeExecutable(contextObject, elementName, arguments);

        observation.forEach(this::reportObservation);
    }

    private List<Map<String, Object>> beforeExecutable(__Gradient_Observable__ contextObject, String elementName, Object[] arguments) {
        assert contextObject != null;
        assert !Strings.isNullOrEmpty(elementName);

        final List<Map<String, Object>> eventList = new ArrayList<>(arguments.length);

        final Map<String, Object> event = createEvent();
        event.put("object_id", contextObject.__gradient_id__());
        event.put("event_type", EventType.CALL.name());
        event.put("element_name", elementName);

        eventList.add(ImmutableMap.copyOf(event));
        eventList.addAll(createParameterEvents(event, elementName, arguments));

        return eventList;
    }

    private List<Map<String, Object>> createParameterEvents(Map<String, Object> baseEvent, String elementName, Object[] arguments) {
        assert baseEvent != null;
        assert elementName != null && !elementName.isEmpty();

        final List<Map<String, Object>> events = new ArrayList<>(arguments.length);
        for (int i = 0; i < arguments.length; i++) {
            final Pair<String, Object> value = normalizeValue(arguments[i]);
            baseEvent.put("event_type", EventType.RECEIVE.name());
            baseEvent.put("element_name", elementName + "#" + i);
            baseEvent.put("value_type", value.getFirst());
            baseEvent.put("value", value.getSecond());

            events.add(ImmutableMap.copyOf(baseEvent));
        }

        return events;
    }

    private Map<String, Object> afterExecutable(__Gradient_Observable__ contextObject, CodeSignature signature, Object returnValue) {
        final String elementName = canonicalExecutableName(signature);

        final Map<String, Object> event = createEvent();
        event.put("object_id", contextObject.__gradient_id__());
        event.put("event_type", Observation.EventType.RETURN.name());
        event.put("element_name", elementName);

        final Pair<String, Object> value = normalizeValue(returnValue);
        event.put("value_type", value.getFirst());
        event.put("value", value.getSecond());

        return event;
    }

    private Pair<String, Object> normalizeValue(Object newState) {

        final Pair<String, Object> result;

        if (newState instanceof Number) {
            result = new Pair<>(GradientType.NUMBER.name(), newState);
        } else if (newState instanceof String) {
            result = new Pair<>(GradientType.TEXT.name(), newState);
        } else if (newState instanceof __Gradient_Observable__) {
            result = new Pair<>(GradientType.REFERENCE.name(), ((__Gradient_Observable__) newState).__gradient_id__());
        } else if (newState == null) {
            result = new Pair<>(GradientType.VOID.name(), "");
        } else {
            result = new Pair<>(GradientType.UNKNOWN.name(), newState.hashCode());
        }

        return result;
    }

    private String canonicalExecutableName(CodeSignature signature) {
        final String declaringType = signature.getDeclaringTypeName();
        final String element = Objects.equals(signature.getName(), "<init>")
                ? signature.getDeclaringType().getSimpleName() : signature.getName();
        final String parameters = Arrays.<Class<?>>stream(signature.getParameterTypes())
                .map(Primitives::wrap)
                .map(Class::getTypeName)
                .collect(Collectors.joining(", "));

        return String.format("%s#%s(%s)", declaringType, element, parameters);
    }

    private Map<String, Object> createEvent() {
        final Map<String, Object> event = new HashMap<>(15);
        event.put("id", Util.INSTANCE.uuid());
        event.put("runtime_id", runtimeId);
        event.put("process_id", processId.get());
        event.put("timestamp", Instant.now().toEpochMilli());
        event.put("context", executableContext.peek());
        event.put("context_element_name", elementNameContext.peek());
        if (executableContext.size() - 2 >= 0) {
            event.put("parent_context", executableContext.get(executableContext.size() - 2));
        }
        if (elementNameContext.size() - 2 >= 0) {
            event.put("parent_context_element_name", elementNameContext.get(elementNameContext.size() - 2));
        }

        return event;
    }

    private void flush() {
        if (structuralCache.present()) {
            behavioralService.report(
                    Objects.requireNonNull(structuralCache.getProjectName()),
                    Objects.requireNonNull(structuralCache.getVersionId()),
                    runtimeId, buffer
            );
        }
        buffer.clear();
    }

    private void reportObservation(Map<String, Object> observation) {
        if (buffer.size() + 1 > gradientConfig.observationCacheSize()) {
            flush();
        }

        buffer.add(observation);
    }
}
