package at.jku.isse.gradient.monitoring;

import at.jku.isse.gradient.GradientConfig;
import at.jku.isse.gradient.dal.Cleanup;
import at.jku.isse.gradient.model.Observation;
import at.jku.isse.gradient.service.BehavioralService;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.primitives.Primitives;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.CodeSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

@Aspect
public class Monitor {

    private static final Logger logger = LoggerFactory.getLogger(Monitor.class);

    private final ThreadLocal<String> processId = ThreadLocal.withInitial(MonitoringUtil::uuid);

    @Inject
    private GradientConfig gradientConfig;

    @Inject
    @Named("runtimeId")
    private String runtimeId;

    @Inject
    private StructuralCache structuralCache;

    @Inject
    private BehavioralService behavioralService;

    @Inject
    @Named("dal.cleanup.bus")
    private EventBus eventBus;

    private final List<Map<String, Object>> buffer = new ArrayList<>();

    private List<String> executableContext = new ArrayList<>();


    Monitor() {
        if (eventBus != null) {
            eventBus.register(this);
        } else {
            logger.warn("Eventbus is not present; hence cleanup may not be possible.");
        }
    }

    @Pointcut(value = "target(contextObject) && set(* *) && args(newState)", argNames = "contextObject,newState")
    void fieldWrite(__Gradient_Observable__ contextObject, Object newState) {
    }

    @Pointcut(value = "target(contextObject) && get(* *)", argNames = "contextObject")
    void fieldRead(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "target(contextObject) && " +
                    "!initialization(at.jku.isse.gradient.monitoring.__Gradient_Observable__.new()) && " +
                    "initialization(*.new(..))",
            argNames = "contextObject")
    void construction(__Gradient_Observable__ contextObject) {
    }

    @Pointcut(value =
            "target(contextObject) && " +
                    "!initialization(at.jku.isse.gradient.monitoring.__Gradient_Observable__.new()) &&" +
                    "!execution(java.lang.String at.jku.isse.gradient.monitoring.__Gradient_Observable__.__gradient_id__()) &&" +
                    "execution(* *.*(..))",
            argNames = "contextObject")
    void invocation(__Gradient_Observable__ contextObject) {
    }


    private ImmutableMap<String, Object> parseObservationValue(String elementName, Object newState) {
        if (newState instanceof Number) {
            return ImmutableMap.of("elementName", elementName, "value", newState, "type", "NUMBER");
        } else if (newState instanceof String) {
            return ImmutableMap.of("elementName", elementName, "value", newState, "type", "TEXT");
        } else if (newState instanceof __Gradient_Observable__) {
            return ImmutableMap.of("elementName", elementName, "value",
                    ((__Gradient_Observable__) newState).__gradient_id__(), "type", "REFERENCE");
        } else {
            if (newState == null) {
                return ImmutableMap.of();
            } else {
                return ImmutableMap.of("elementName", elementName, "value", newState.hashCode(), "type", "UNKNOWN");
            }
        }
    }

    @After(value = "fieldWrite(contextObject, newState)", argNames = "thisJoinPointStaticPart,contextObject,newState")
    public void fieldWriteAdvice(JoinPoint.StaticPart thisJoinPointStaticPart,
                                 __Gradient_Observable__ contextObject, Object newState) {
        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", MonitoringUtil.uuid())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.WRITE.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableList<ImmutableMap<String, Object>> values = ImmutableList.of(parseObservationValue(elementName, newState));
        if (!values.isEmpty()) {
            observation.put("writes", values);
        }

        reportObservation(observation.build());
    }


    @AfterReturning(pointcut = "fieldRead(contextObject)", returning = "field",
            argNames = "thisJoinPointStaticPart,contextObject,field")
    public void fieldReadAdvice(JoinPoint.StaticPart thisJoinPointStaticPart,
                                __Gradient_Observable__ contextObject,
                                Object field) {
        final Signature signature = thisJoinPointStaticPart.getSignature();
        final String elementName = signature.getDeclaringTypeName() + "#" + signature.getName();

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", MonitoringUtil.uuid())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.READ.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));
        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableList<ImmutableMap<String, Object>> values = ImmutableList.of(parseObservationValue(elementName, field));
        if (!values.isEmpty()) {
            observation.put("reads", values);
        }

        reportObservation(observation.build());
    }


    @After(value = "construction(contextObject)", argNames = "thisJoinPoint,thisJoinPointStaticPart,contextObject")
    public void constructionAdvice(JoinPoint thisJoinPoint,
                                   JoinPoint.StaticPart thisJoinPointStaticPart,
                                   __Gradient_Observable__ contextObject) {
        callAdvice(thisJoinPoint, thisJoinPointStaticPart, contextObject);
    }

    @Before(value = "invocation(contextObject)", argNames = "thisJoinPoint,thisJoinPointStaticPart,contextObject")
    public void invocationBeforeAdvice(JoinPoint thisJoinPoint,
                                       JoinPoint.StaticPart thisJoinPointStaticPart,
                                       __Gradient_Observable__ contextObject) {
        callAdvice(thisJoinPoint, thisJoinPointStaticPart, contextObject);
    }


    @AfterReturning(value = "invocation(contextObject)", returning = "returnValue", argNames = "thisJoinPointStaticPart,contextObject,returnValue")
    public void invocationAfterAdvice(JoinPoint.StaticPart thisJoinPointStaticPart,
                                      __Gradient_Observable__ contextObject, Object returnValue) {
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();

        final Map<String, Object> observation = afterExecutable(contextObject, signature, returnValue);

        executableContext.remove(executableContext.size() - 1);

        reportObservation(observation);
    }

    @SuppressWarnings("unused")
    public void reset() {
        buffer.clear();

        if (structuralCache != null) {
            structuralCache.reload();
        } else {
            logger.warn("Could not fully reset the aspect as it was not proper initialized.");
        }
    }

    @Subscribe
    public void cleanupSignal(Cleanup cleanup) {
        flush();
        if (behavioralService != null && structuralCache != null) {
            behavioralService.reportObservationEnd(structuralCache.getProjectName(), structuralCache.getVersionId(), runtimeId);
        } else {
            logger.warn("Could not cleanup as the aspectj was not fully initialized.");
        }
    }

    private void callAdvice(JoinPoint thisJoinPoint, JoinPoint.StaticPart thisJoinPointStaticPart, __Gradient_Observable__ contextObject) {
        assert thisJoinPointStaticPart.getSignature() instanceof CodeSignature : "Expected method selected via pointcut";

        executableContext.add(MonitoringUtil.uuid());
        final CodeSignature signature = (CodeSignature) thisJoinPointStaticPart.getSignature();
        final Object[] arguments = thisJoinPoint.getArgs();

        final Map<String, Object> observation = beforeExecutable(contextObject, signature, arguments);

        reportObservation(observation);
    }

    private Map<String, Object> beforeExecutable(__Gradient_Observable__ contextObject, CodeSignature signature, Object[] arguments) {
        final String[] parameters = signature.getParameterNames();
        final String elementName = canonicalExecutableName(signature);
        assert arguments.length == parameters.length;


        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", MonitoringUtil.uuid())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.CALL.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final List<Map<String, Object>> values = new ArrayList<>(arguments.length);
        for (int i = 0; i < arguments.length; i++) {
            values.add(parseObservationValue(elementName + "#" + i, arguments[i]));
        }

        if (!values.isEmpty()) {
            observation.put("receives", values);
        }

        return observation.build();
    }

    private Map<String, Object> afterExecutable(__Gradient_Observable__ contextObject, CodeSignature signature, Object returnValue) {
        String elementName = canonicalExecutableName(signature);

        final ImmutableMap.Builder<String, Object> observation = ImmutableMap.<String, Object>builder()
                .put("id", MonitoringUtil.uuid())
                .put("processId", processId.get())
                .put("objectId", contextObject.__gradient_id__())
                .put("timestamp", Instant.now().toEpochMilli())
                .put("observeeName", elementName)
                .put("type", Observation.EventType.RETURN.name())
                .put("executableContext", executableContext.get(executableContext.size() - 1));

        if (executableContext.size() - 2 > -1) {
            observation.put("parentExecutableContext", executableContext.get(executableContext.size() - 2));
        }

        final ImmutableMap<String, Object> value = parseObservationValue(elementName, returnValue);
        if (!value.isEmpty()) {
            observation.put("returns", ImmutableList.of(value));
        }

        return observation.build();
    }

    private String canonicalExecutableName(CodeSignature signature) {
        final String declaringType = signature.getDeclaringTypeName();
        final String element = Objects.equals(signature.getName(), "<init>")
                ? signature.getDeclaringType().getSimpleName() : signature.getName();
        final String parameters = Arrays.<Class<?>>stream(signature.getParameterTypes())
                .map(Primitives::wrap)
                .map(Class::getTypeName)
                .collect(Collectors.joining(", "));

        return String.format("%s#%s(%s)", declaringType, element, parameters);
    }

    private void flush() {
        behavioralService.report(structuralCache.getProjectName(), structuralCache.getVersionId(), runtimeId, buffer);
        buffer.clear();
    }

    private boolean isInitialized() {
        return runtimeId != null && structuralCache != null && behavioralService != null;
    }

    private boolean reportErrorIssued = false;

    private void reportObservation(Map<String, Object> observation) {
        if (isInitialized()) {
            if (buffer.size() + 1 > gradientConfig.observationCacheSize()) {
                flush();
            }
        } else {
            if (!reportErrorIssued) {
                logger.error("Services are not initialized; hence observations will be discarded.");
                reportErrorIssued = true;
            }
            buffer.clear();
        }

        buffer.add(observation);
    }
}
