/*
 * Copyright (c) 2017. Hans-Peter Grahsl (grahslhp@gmail.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package at.grahsl.kafka.connect.mongodb.cdc.debezium.rdbms.outbox;

import at.grahsl.kafka.connect.mongodb.MongoDbSinkConnectorConfig;
import at.grahsl.kafka.connect.mongodb.cdc.CdcOperation;
import at.grahsl.kafka.connect.mongodb.cdc.debezium.DebeziumCdcHandler;
import at.grahsl.kafka.connect.mongodb.cdc.debezium.OperationType;
import at.grahsl.kafka.connect.mongodb.cdc.debezium.rdbms.RdbmsDelete;
import at.grahsl.kafka.connect.mongodb.cdc.debezium.rdbms.RdbmsInsert;
import at.grahsl.kafka.connect.mongodb.cdc.debezium.rdbms.RdbmsUpdate;
import at.grahsl.kafka.connect.mongodb.converter.SinkDocument;
import com.mongodb.DBCollection;
import com.mongodb.client.model.WriteModel;
import org.apache.kafka.connect.errors.DataException;
import org.bson.BsonDocument;
import org.bson.BsonInvalidOperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

public class OutboxHandler extends DebeziumCdcHandler {

    public static final String JSON_DOC_BEFORE_FIELD = "before";
    public static final String JSON_DOC_AFTER_FIELD = "after";
    public static final String JSON_DOC_PAYLOAD_FIELD = "payload";
    public static final String PAYLOAD_DEFAULT_ID_FIELD = "id";

    private static Logger logger = LoggerFactory.getLogger(OutboxHandler.class);

    public OutboxHandler(MongoDbSinkConnectorConfig config) {
        super(config);
        final Map<OperationType,CdcOperation> operations = new HashMap<>();
        operations.put(OperationType.CREATE,new RdbmsInsert());
        operations.put(OperationType.READ,new RdbmsInsert());
        operations.put(OperationType.UPDATE,new RdbmsUpdate());
        operations.put(OperationType.DELETE,new RdbmsDelete());
        registerOperations(operations);
    }

    public OutboxHandler(MongoDbSinkConnectorConfig config,
                        Map<OperationType,CdcOperation> operations) {
        super(config);
        registerOperations(operations);
    }

    @Override
    public Optional<WriteModel<BsonDocument>> handle(SinkDocument doc) {

        BsonDocument keyDoc = doc.getKeyDoc().orElseGet(BsonDocument::new);

        BsonDocument valueDoc = doc.getValueDoc().orElseGet(BsonDocument::new);

        if (valueDoc.isEmpty())  {
            logger.debug("skipping debezium tombstone event for kafka topic compaction");
            return Optional.empty();
        }

        //TODO & FIXME: temp hack to check if "id" is contained then it is considered an aggregate root
        // try to do this based on (event type + aggregatetype) filtering instead???

        CdcOperation op = getCdcOperation(valueDoc);
        if(op instanceof OutboxInsert) {
            BsonDocument payload = extractAndParseBsonDocumentFromPayloadField(valueDoc);
            if(!payload.containsKey(PAYLOAD_DEFAULT_ID_FIELD)) {
                logger.debug("skipping debezium cdc outbox event which is missing the default 'id' field and is thus most likely not an aggregate root");
                return Optional.empty();
            }
        }

        return Optional.ofNullable(getCdcOperation(valueDoc)
                            .perform(new SinkDocument(keyDoc,valueDoc)));
    }

    protected static BsonDocument extractAndParseBsonDocumentFromPayloadField(BsonDocument valueDoc) {
        try {
            BsonDocument payload = BsonDocument.parse(valueDoc.getDocument(JSON_DOC_AFTER_FIELD)
                    .getString(JSON_DOC_PAYLOAD_FIELD)
                    .getValue());

            if (payload.isEmpty())
                throw new BsonInvalidOperationException("after contains an empty payload field");
            return payload;
        } catch (BsonInvalidOperationException exc) {
            throw new DataException("error: value doc 'payload' field is empty or has invalid type" +
                    " for insert/read operation which seems severely wrong -> defensive actions taken!", exc);
        }
    }

    protected static BsonDocument generateFilterDoc(BsonDocument keyDoc, BsonDocument valueDoc, OperationType opType) {
        //NOTE: for outbox events this CDC handler ignores the key part which is a UUID for deduplication
        //when focusing on the payload part and using UPSERT semantic towards the sink nothing needs
        //to be done in this regard
        if (opType.equals(OperationType.CREATE) || opType.equals(OperationType.READ)) {
            //no before state -> filter doc parsed from payload field 'id' in after
            BsonDocument payload = extractAndParseBsonDocumentFromPayloadField(valueDoc);
            //TODO: what if the id field is not called 'id' ??? configure this somewhere but for starters
            //stick to "default" naming conventions in the DBZ reference implementation


            return new BsonDocument(DBCollection.ID_FIELD_NAME, new BsonDocument(PAYLOAD_DEFAULT_ID_FIELD, payload.get(PAYLOAD_DEFAULT_ID_FIELD)));
        }
        if (opType.equals(OperationType.UPDATE)) {
            throw new DataException("error: operation type "+OperationType.UPDATE + " unexpected and not allowed for outbox events");
        }
        if (opType.equals(OperationType.DELETE)) {
            throw new DataException("error: operation type "+OperationType.DELETE + " must not be handled for outbox events - a no op should be configured for this");
        }

        throw new DataException("error: unknown or invalid operation type "+opType);
    }

    protected static BsonDocument generateUpsertOrReplaceDoc(BsonDocument keyDoc, BsonDocument valueDoc, BsonDocument filterDoc) {

        BsonDocument payloadDoc = extractAndParseBsonDocumentFromPayloadField(valueDoc);

        BsonDocument upsertDoc = new BsonDocument();
        if(filterDoc.containsKey(DBCollection.ID_FIELD_NAME)) {
            upsertDoc.put(DBCollection.ID_FIELD_NAME,filterDoc.get(DBCollection.ID_FIELD_NAME));
        }

        //TODO: filter out the payload "id" field which is (part of) the key already
        payloadDoc.keySet().forEach(f -> upsertDoc.put(f,payloadDoc.get(f)));

        return upsertDoc;
    }

}
