package at.florianschuster.androidreactor

import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.functions.Consumer


/**
 * Consumes a [Reactor] Action with the corresponding [Reactor].
 */
fun <Action : Any> Observable<Action>.consume(with: Reactor<in Action, *, *>): Disposable =
    subscribe(with.action::accept, AndroidReactor::handleError)


/**
 * Maps changes from a [State] Observable and only emits those that are distinct from their immediate predecessors.
 */
fun <State : Any, SubState : Any> Observable<State>.changesFrom(mapper: (State) -> SubState): Observable<out SubState> {
    return map(mapper::invoke)
        .distinctUntilChanged()
}


/**
 * Binds an observable to the UI. Also logs errors in [AndroidReactor].
 */
fun <State : Any> Observable<State>.bind(): Disposable {
    return observeOn(AndroidSchedulers.mainThread())
        .doOnError(AndroidReactor::handleError)
        .subscribe()
}

/**
 * Binds an observable to a UI target. Also logs errors in [AndroidReactor].
 */
fun <State : Any> Observable<State>.bind(to: Consumer<in State>): Disposable {
    return observeOn(AndroidSchedulers.mainThread())
        .subscribe(to, Consumer(AndroidReactor::handleError))
}

/**
 * Binds an observable to a UI target. Also logs errors in [AndroidReactor].
 */
fun <State : Any> Observable<State>.bind(to: (State) -> Unit): Disposable {
    return observeOn(AndroidSchedulers.mainThread())
        .subscribe(to, AndroidReactor::handleError)
}