package at.florianschuster.androidreactor

import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.functions.Consumer


/**
 * Consumes a Reactor Action with the corresponding Reactor.
 */
fun <Action : Any> Observable<Action>.consume(with: Reactor<in Action, *, *>): Disposable =
    subscribe(with.action::accept, AndroidReactor::log)


/**
 * Maps changes from a State Observable and only emits those that are distinct from their immediate predecessors.
 */
fun <State : Any, SubState : Any> Observable<State>.changesFrom(mapper: (State) -> SubState): Observable<out SubState> {
    return map(mapper::invoke)
        .distinctUntilChanged()
}


/**
 * Binds an observable to a UI target. Also logs Errors in AndroidReactor.
 */
fun <State : Any> Observable<State>.bind(to: Consumer<in State>): Disposable =
    bind(to::accept)

/**
 * Binds an observable to a UI target. Also logs Errors in AndroidReactor.
 */
fun <State : Any> Observable<State>.bind(to: (State) -> Unit): Disposable {
    return observeOn(AndroidSchedulers.mainThread())
        .subscribe(to, AndroidReactor::log)
}