/*
 * Decompiled with CFR 0.152.
 */
package at.favre.lib.bytes;

import at.favre.lib.bytes.Bytes;
import at.favre.lib.bytes.Util;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Random;

public interface BytesTransformer {
    public byte[] transform(byte[] var1, boolean var2);

    public static class BitSwitchTransformer
    implements BytesTransformer {
        private final int position;
        private final Boolean newBitValue;

        BitSwitchTransformer(int position, Boolean newBitValue) {
            this.position = position;
            this.newBitValue = newBitValue;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            byte[] out;
            byte[] byArray = out = inPlace ? currentArray : Bytes.from(currentArray).array();
            if (this.position < 0 || this.position >= 8 * currentArray.length) {
                throw new IllegalArgumentException("bit index " + this.position * 8 + " out of bounds");
            }
            int bytePosition = currentArray.length - 1 - this.position / 8;
            if (this.newBitValue == null) {
                int n = bytePosition;
                out[n] = (byte)(out[n] ^ 1 << this.position % 8);
            } else if (this.newBitValue.booleanValue()) {
                int n = bytePosition;
                out[n] = (byte)(out[n] | 1 << this.position % 8);
            } else {
                int n = bytePosition;
                out[n] = (byte)(out[n] & ~(1 << this.position % 8));
            }
            return out;
        }
    }

    public static final class ResizeTransformer
    implements BytesTransformer {
        private final int newSize;

        ResizeTransformer(int newSize) {
            this.newSize = newSize;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            if (currentArray.length == this.newSize) {
                return currentArray;
            }
            if (this.newSize < 0) {
                throw new IllegalArgumentException("cannot resize to smaller than 0");
            }
            if (this.newSize == 0) {
                return new byte[0];
            }
            byte[] resizedArray = new byte[this.newSize];
            if (this.newSize > currentArray.length) {
                System.arraycopy(currentArray, 0, resizedArray, Math.max(0, Math.abs(this.newSize - currentArray.length)), Math.min(this.newSize, currentArray.length));
            } else {
                System.arraycopy(currentArray, Math.max(0, Math.abs(this.newSize - currentArray.length)), resizedArray, Math.min(0, Math.abs(this.newSize - currentArray.length)), Math.min(this.newSize, currentArray.length));
            }
            return resizedArray;
        }
    }

    public static final class CopyTransformer
    implements BytesTransformer {
        final int offset;
        final int length;

        CopyTransformer(int offset, int length) {
            this.offset = offset;
            this.length = length;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            byte[] copy = new byte[this.length];
            System.arraycopy(currentArray, this.offset, copy, 0, copy.length);
            return copy;
        }
    }

    public static final class ShuffleTransformer
    implements BytesTransformer {
        private final Random random;

        ShuffleTransformer(Random random) {
            Objects.requireNonNull(random, "passed random must not be null");
            this.random = random;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            byte[] out = inPlace ? currentArray : Bytes.from(currentArray).array();
            Util.shuffle(out, this.random);
            return out;
        }
    }

    public static final class SortTransformer
    implements BytesTransformer {
        private final Comparator<Byte> comparator;

        SortTransformer() {
            this(null);
        }

        SortTransformer(Comparator<Byte> comparator) {
            this.comparator = comparator;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            if (this.comparator == null) {
                byte[] out = inPlace ? currentArray : Bytes.from(currentArray).array();
                Arrays.sort(out);
                return out;
            }
            List<Byte> list = Bytes.wrap(currentArray).toList();
            Collections.sort(list, this.comparator);
            return Bytes.from(list).array();
        }
    }

    public static final class ReverseTransformer
    implements BytesTransformer {
        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            byte[] out = inPlace ? currentArray : Bytes.from(currentArray).array();
            for (int i = 0; i < out.length / 2; ++i) {
                byte temp = out[i];
                out[i] = out[out.length - i - 1];
                out[out.length - i - 1] = temp;
            }
            return out;
        }
    }

    public static final class ConcatTransformer
    implements BytesTransformer {
        private final byte[] secondArray;

        ConcatTransformer(byte[] secondArrays) {
            Objects.requireNonNull(secondArrays, "the second byte array must not be null");
            this.secondArray = secondArrays;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            return Util.concat(currentArray, this.secondArray);
        }
    }

    public static final class ShiftTransformer
    implements BytesTransformer {
        private final int shiftCount;
        private final Type type;

        ShiftTransformer(int shiftCount, Type type) {
            Objects.requireNonNull(type, "passed shift type must not be null");
            this.shiftCount = shiftCount;
            this.type = type;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            BigInteger bigInt = new BigInteger(currentArray);
            switch (this.type) {
                default: {
                    return bigInt.shiftLeft(this.shiftCount).toByteArray();
                }
                case RIGHT_SHIFT: 
            }
            return bigInt.shiftRight(this.shiftCount).toByteArray();
        }

        static enum Type {
            LEFT_SHIFT,
            RIGHT_SHIFT;

        }
    }

    public static final class NegateTransformer
    implements BytesTransformer {
        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            byte[] out = inPlace ? currentArray : Bytes.from(currentArray).array();
            for (int i = 0; i < out.length; ++i) {
                out[i] = ~out[i];
            }
            return out;
        }
    }

    public static final class BitWiseOperatorTransformer
    implements BytesTransformer {
        private final byte[] secondArray;
        private final Mode mode;

        BitWiseOperatorTransformer(byte[] secondArray, Mode mode) {
            Objects.requireNonNull(secondArray, "the second byte array must not be null");
            Objects.requireNonNull(mode, "passed bitwise mode must not be null");
            this.secondArray = secondArray;
            this.mode = mode;
        }

        @Override
        public byte[] transform(byte[] currentArray, boolean inPlace) {
            if (currentArray.length != this.secondArray.length) {
                throw new IllegalArgumentException("all byte array must be of same length doing bit wise operation");
            }
            byte[] out = inPlace ? currentArray : new byte[currentArray.length];
            block4: for (int i = 0; i < currentArray.length; ++i) {
                switch (this.mode) {
                    default: {
                        out[i] = (byte)(currentArray[i] | this.secondArray[i]);
                        continue block4;
                    }
                    case AND: {
                        out[i] = (byte)(currentArray[i] & this.secondArray[i]);
                        continue block4;
                    }
                    case XOR: {
                        out[i] = (byte)(currentArray[i] ^ this.secondArray[i]);
                    }
                }
            }
            return out;
        }

        static enum Mode {
            AND,
            OR,
            XOR;

        }
    }
}

