package at.creadoo.homer.shell.presence.commands;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.karaf.shell.api.action.Command;
import org.apache.karaf.shell.api.action.Option;
import org.apache.karaf.shell.api.action.lifecycle.Service;
import org.joda.time.LocalTime;

import com.google.common.collect.TreeBasedTable;

import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.creadoo.homer.processing.presence.Constants;
import at.creadoo.homer.processing.presence.model.ActuatorItem;
import at.creadoo.homer.shell.presence.commands.util.ConsoleUtil;

/**
 * Command to list all scheduled items.
 */
@Command(scope = "presence", name = "list", description = "List scheduled items.")
@Service
public class PresenceList extends CommandSupport {

    protected static final String LIST_HEADER_FORMAT = "%-8s %-25s %-15s %-15s";
    protected static final String LIST_OUTPUT_FORMAT = "%-8s %-25s %-15s %-15s";

    protected static final String VISUAL_OUTPUT_FORMAT = "%-3s %-15s %s";

    protected static final String UNKNOWN = "---";
    protected static final String ITEM_DEVICE = "Device";
    protected static final String ITEM_MESSAGE = "Message";
    protected static final String ITEM_START_TIME = "Start Time";
    protected static final String ITEM_END_TIME = "End Time";

    @Option(required = false, name = "-v", aliases = "--visual", description = "Enable the visual mode.", multiValued = false)
    protected boolean visualMode = false;

    @Option(required = false, name = "-d", aliases = "--visual-detailed", description = "Enable the detailed visual mode.", multiValued = false)
    protected boolean visualModeDetailed = false;

    @Option(required = false, name = "-p", aliases = "--visual-detailed-parts", description = "Parts for the detailed visual mode. A value between 1 and 24.", multiValued = false)
    protected int visualModeDetailedParts = 4;

    @Override
	public Object execute() throws Exception {
		if (getPresenceSimulator() != null) {
			final List<ActuatorItem> actuatorItems = getPresenceSimulator().getActuatorItems();
			if (!visualMode) {
				getConsole().println(String.format(LIST_HEADER_FORMAT, ITEM_DEVICE, ITEM_MESSAGE, ITEM_START_TIME, ITEM_END_TIME));
	
				Collections.sort(actuatorItems, new SortActuatorItemByDeviceId());
				
				for (ActuatorItem item : actuatorItems) {
					getConsole().println(String.format(LIST_OUTPUT_FORMAT, item.getDevice().getActuatorId(), item.getDeviceMessageType() == null ? UNKNOWN : item.getDeviceMessageType().getTitle(), item.getStartTime() == null ? UNKNOWN : TIME_FORMAT.print(item.getStartTime()), item.getEndTime() == null ? UNKNOWN : TIME_FORMAT.print(item.getEndTime())));
				}
			} else {
				final Integer width = ConsoleUtil.getConsoleSize(getSession()).getWidth() - 1;
				final Integer widthLeftover = width - String.format(VISUAL_OUTPUT_FORMAT, "", "", "").length();
				
				final TreeBasedTable<Integer, DeviceMessageType, List<ActuatorItem>> items = TreeBasedTable.create();
				
				// Fill table structure
				for (ActuatorItem item : actuatorItems) {
					if (!items.contains(item.getDevice().getActuatorId(), item.getDeviceMessageType())) {
						items.put(item.getDevice().getActuatorId(), item.getDeviceMessageType(), new ArrayList<ActuatorItem>());
					}
					
					final List<ActuatorItem> valueList = items.get(item.getDevice().getActuatorId(), item.getDeviceMessageType());
					valueList.add(item);
				}
				
				if (!visualModeDetailed) {
					final LocalTime startTime = Constants.parseTime("00:00");
					final LocalTime endTime = startTime.plusSeconds(Constants.SECONDS_PER_DAY).minusMillis(1);
					
					showVisualModePart(widthLeftover, items.values(), startTime, endTime);
					getConsole().println(StringUtils.repeat("-", width));
				} else {
					int parts = 4;
					if (visualModeDetailedParts > 0 && visualModeDetailedParts <= 24) {
						parts = visualModeDetailedParts;
					}
					for (int i = 0; i < parts; i++) {
						final LocalTime startTime = Constants.parseTime("00:00").plusSeconds(Constants.SECONDS_PER_DAY / parts * i);
						final LocalTime endTime = startTime.plusSeconds(Constants.SECONDS_PER_DAY / parts).minusMillis(1);
						
						showVisualModePart(widthLeftover, items.values(), startTime, endTime);
						getConsole().println(StringUtils.repeat("-", width));
					}
				}
			}
		} else {
			printError(ERROR_PRESENCE_SIMULATOR);
		}
        
        return null;
    }
    
    private void showVisualModePart(final int length, final Collection<List<ActuatorItem>> items, final LocalTime startTime, final LocalTime endTime) {
		// Print header
		getConsole().println(String.format(VISUAL_OUTPUT_FORMAT, "", "", calculateVisualModeHeader(length, startTime, endTime)));
		
		// Print items
		for (List<ActuatorItem> valueList : items) {
			if (valueList.size() > 0) {
				final ActuatorItem item = valueList.get(0);
				getConsole().println(String.format(VISUAL_OUTPUT_FORMAT, item.getDevice().getActuatorId(), item.getDeviceMessageType() == null ? UNKNOWN : item.getDeviceMessageType().getTitle(), calculateVisualModeItem(length, valueList, startTime, endTime)));
			}
		}
    }
    
    private String calculateVisualModeHeader(final int length, final LocalTime startTime, final LocalTime endTime) {
    	final String result = StringUtils.rightPad(Constants.TIME_FORMAT.print(startTime), length - Constants.TIME_FORMAT.print(startTime).length()) + Constants.TIME_FORMAT.print(endTime);
    	return result;
    }
    
    private String calculateVisualModeItem(final int length, final List<ActuatorItem> items, final LocalTime startTime, final LocalTime endTime) {
    	final int lengthContent = length - 2;
    	
    	final char[] result = new char[lengthContent];
    	for (int i = 0; i < lengthContent; i++) {
    		final Integer startMillis = startTime.getMillisOfDay() + ((endTime.getMillisOfDay() - startTime.getMillisOfDay()) / lengthContent) * i;
    		final Integer endMillis = startTime.getMillisOfDay() + ((endTime.getMillisOfDay() - startTime.getMillisOfDay()) / lengthContent) * (i + 1);
    		
    		result[i] = ' ';
    		
    		for (ActuatorItem item : items) {
    			// start and end in interval
    			if (item.getStartTime().getMillisOfDay() >= startMillis && item.getEndTime().getMillisOfDay() <= endMillis) {
    				result[i] = 'x';
    				continue;
    			// start and end out of interval
    			} else if (item.getStartTime().getMillisOfDay() <= startMillis && item.getEndTime().getMillisOfDay() >= endMillis) {
    				result[i] = 'X';
    				continue;
    				// start in interval and end out of interval
    			} else if (item.getStartTime().getMillisOfDay() >= startMillis  && item.getStartTime().getMillisOfDay() < endMillis) {
    				result[i] = '.';
    				continue;
    				// start out of interval and end in interval
    			} else if (item.getEndTime().getMillisOfDay() > startMillis  && item.getEndTime().getMillisOfDay() <= endMillis) {
    				result[i] = '.';
    				continue;
    			}
    		}
    	}
    	
    	return "|" + new String(result) + "|";
    }
    
	private class SortActuatorItemByDeviceId implements Comparator<ActuatorItem> {
		@Override
		public int compare(final ActuatorItem item1, final ActuatorItem item2) {
			int result = item1.getDevice().getActuatorId().compareTo(item2.getDevice().getActuatorId());
			if (result == 0) {
				result = item1.getDeviceMessageType().compareTo(item2.getDeviceMessageType());
				if (result == 0) {
					result = item1.getStartTime().compareTo(item2.getStartTime());
					if (result == 0) {
						result = item1.getEndTime().compareTo(item2.getEndTime());
					}
				}
			}
			return result;
		}
	}

}
